﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace arinc429_fifo_WinForms_gui
{
    using U64BIT = System.UInt64;
    using U32BIT = System.UInt32;
    using U16BIT = System.UInt16;
    using U8BIT = System.Byte;
    using S64BIT = System.Int64;
    using S32BIT = System.Int32;
    using S16BIT = System.Int16;
    using S8BIT = System.SByte;

    /* Channel Count struct */
    public struct CHANCOUNT_t
    {
        public U8BIT bTx;
        public U8BIT bRx;
        public U8BIT bGroup;
        public U8BIT bDiscrete;
        public U8BIT bAvionic;
        public U8BIT bBoardModel;
        public U8BIT b1553;
        public U8BIT a429Prog;
        public U8BIT bUart;
        public U8BIT RS232;
        public U8BIT RS485;
        public U8BIT CanBus;
        public U8BIT a717Rx;
        public U8BIT a717Tx;
        public U8BIT a717Prog;
        public U8BIT bDioTt;
    }

    public struct RX_HBUF_MESSAGE
    {
        public U32BIT u32Data;
        public U32BIT u32StampHigh;
        public U32BIT u32StampLow;
    }

    public enum Speed { DD429_LOW_SPEED = 0, DD429_HIGH_SPEED, DD429_ARINC575 };
    public enum Parity { DD429_NO_PARITY = 0, DD429_ODD_PARITY, DD429_EVEN_PARITY };

    public enum TimeStampFormat
    {
        TT48 = 0,
        IRIG_A,
        IRIG_B,
        IRIG_C,
        IRIG_D,
        IRIG_B_ENHANCED,
        MAX_TTFORMAT
    }

    public enum TimeStampRollover
    {
        TTRO_16 = 0,
        TTRO_17,
        TTRO_18,
        TTRO_19,
        TTRO_20,
        TTRO_21,
        TTRO_22,
        TTRO_48,
        MAX_TTRO
    }

    public enum TimeStampResolution
    {
        TTRES_64 = 0,
        TTRES_32,
        TTRES_16,
        TTRES_8,
        TTRES_4,
        TTRES_2,
        TTRES_1,
        TTRES_500,
        TTRES_100,
        MAX_TTRES
    };

    public partial class arinc429_fifo_form : Form
    {
        #region Imported API from DDC SDK Dll
            [DllImport("DD42992.dll")]
            static extern S16BIT InitCard(
                S16BIT s16Card);

            [DllImport("DD42992.dll")]
            static extern S16BIT GetChannelCount(
                S16BIT s16Card,
                ref CHANCOUNT_t refChanCount);

            [DllImport("DD42992.dll")]
            static extern S16BIT FreeCard(
                S16BIT s16Card);

            [DllImport("DD42992.dll")]
            static extern S16BIT EnableTx(
                S16BIT s16Card,
                S16BIT s16Transmitter,
                S16BIT s16Enable
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT SetTxSpeed(
                S16BIT s16Card,
                S16BIT s16Transmitter,
                S16BIT s16Speed
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT LoadTxQueueOne(
                S16BIT s16Card,
                S16BIT s16Transmitter,
                U32BIT u32Data
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT InstallFifoRxHostBuffer(
                S16BIT s16Card,
                U32BIT u32Reserved
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT SetRxChannelSpeed(
                S16BIT s16Card,
                S16BIT s16Receiver,
                S16BIT s16Speed
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT SetRxChannelParity(
                S16BIT s16Card,
                S16BIT s16Receiver,
                S16BIT s16Parity
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT SetRxChannelMode(
                S16BIT s16Card,
                S16BIT s16Receiver,
                S16BIT s16Mode
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT EnableRxHostBuffer(
                S16BIT s16Card,
                S16BIT s16Receiver,
                S16BIT u32Reserved
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT EnableFilter(
                S16BIT s16Card,
                S16BIT s16Receiver,
                S16BIT s16Enable
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT EnableRx(
                S16BIT s16Card,
                S16BIT s16Receiver,
                S16BIT s16Enable
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT DisableRxHostBuffer(
                S16BIT s16Card,
                S16BIT s16Receiver
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT ResetRxChannel(
                S16BIT s16Card,
                S16BIT s16Receiver
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT UninstallFifoRxHostBuffer(
                S16BIT s16Card
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT ReadRxHostBuffer(
                S16BIT s16Card,
                S16BIT s16Receiver,
                U16BIT u16NumMsgsToRead,
                ref U16BIT u16NumMsgsRead,
                [Out] RX_HBUF_MESSAGE[] sMessages
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT ConfigTimeStamp(
                S16BIT s16Card,
                U8BIT u8Format,
                U8BIT u8Rollover,
                U8BIT u8Resolution
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT ResetTimeStamp(
                S16BIT s16Card
                );

            [DllImport("DD42992.dll")]
            static extern S16BIT EnableTimeStamp(
                S16BIT s16Card,
                S16BIT s16Receiver,
                S16BIT s16Enable
            );
        #endregion

        /* define DDC SDK constants */
        public const S16BIT ERR_SUCCESS = 0;
        public const S16BIT DD429_DISABLE = 0;
        public const S16BIT DD429_ENABLE = 1;
        public const S16BIT FIFO_MODE = 1;
        public const S16BIT ERR_OVERFLOW = -200;
        public const S16BIT ERR_RX_HBUF_OVERFLOW = -222;
        public const U16BIT MAX_NUM_MESSAGES_TO_READ = 3069;

        U32BIT[] au32Data = new U32BIT[]
                            {
                                0x043210A6, 0x04321054, 0x04321037, 0x04321055, 0x04321066
                            };

        CHANCOUNT_t sChanCount = new CHANCOUNT_t();
        S16BIT s16Result, s16Card;
        U8BIT u8Chan, u8NumTx, u8NumRx;
        RX_HBUF_MESSAGE[] sRxHbufMessages = new RX_HBUF_MESSAGE[MAX_NUM_MESSAGES_TO_READ];


        private void button_FreeCard_Click(object sender, EventArgs e)
        {
            comboBox_TxChannel.Items.Clear();
            comboBox_RxChannel.Items.Clear();

            s16Card = Int16.Parse(comboBox_CardNum.SelectedItem.ToString());

            s16Result = UninstallFifoRxHostBuffer(s16Card);
            if (s16Result != ERR_SUCCESS)
            {
                MessageBox.Show("UninstallFifoRxHostBuffer Failed");
                return;
            }
            else
            {
                // MessageBox.Show("UninstallFifoRxHostBuffer success!");
            }

            /* Free device */
            s16Result = FreeCard(s16Card);
            if (s16Result != ERR_SUCCESS)
            {
                MessageBox.Show("FreeCard Failed");
                return;
            }
            else
            {
                MessageBox.Show("FreeCard success!");
                button_InitCard.Enabled = true;
                button_FreeCard.Enabled = false;
                button_EnableRx.Enabled = false;
                button_DisableRx.Enabled = false;
                button_EnableTx.Enabled = false;
                button_DisableTx.Enabled = false;
                button_Tx.Enabled = false;
                button_Rx1Msg.Enabled = false;  
                button_RxAllMsgs.Enabled = false;   
                textBoxHexData.Enabled = false;
            }
        }

        private void button_InitCard_Click(object sender, EventArgs e)
        {
            comboBox_TxChannel.Items.Clear();
            comboBox_RxChannel.Items.Clear();

            s16Card = Int16.Parse(comboBox_CardNum.SelectedItem.ToString());

            /* Initialize device */
            s16Result = InitCard(s16Card);
            if (s16Result != ERR_SUCCESS)
            {
                MessageBox.Show("InitCard Failed", "ARINC 429 Card Initialisation", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            else
            {
                // MessageBox.Show("InitCard success!");
            }

            /* Configure time stamp feature to 48 bit, roll over at max. value, and 100ns resolution */
            s16Result = ConfigTimeStamp(s16Card, (U8BIT)TimeStampFormat.TT48, (U8BIT)TimeStampRollover.TTRO_48, (U8BIT)TimeStampResolution.TTRES_100/*TTRES_1*/);
            if (s16Result != ERR_SUCCESS)
            {
                MessageBox.Show("ConfigTimeStamp Failed", "ARINC 429 Card Initialisation", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            else
            {
                // MessageBox.Show("ConfigTimeStamp success!");
            }

            /* Clear timer */
            s16Result = ResetTimeStamp(s16Card);
            if (s16Result != ERR_SUCCESS)
            {
                MessageBox.Show("ResetTimeStamp Failed", "ARINC 429 Card Initialisation", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            else
            {
                // MessageBox.Show("ResetTimeStamp success!");
            }

            /* Determine device channel count */
            s16Result = GetChannelCount(s16Card, ref sChanCount);
            if (s16Result != ERR_SUCCESS)
            {
                MessageBox.Show("GetChannelCount Failed", "ARINC 429 Card Initialisation", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            else
            {
                // MessageBox.Show("GetChannelCount success!");
                u8NumTx = sChanCount.bTx;
                u8NumRx = sChanCount.bRx;

                /* If no fixed tx channels, check for existance of programmable */
                if (u8NumTx == 0)
                {
                    MessageBox.Show("ARINC 429 card has no fixed tx channels!");

                    if (sChanCount.a429Prog > 0)
                        u8NumTx = sChanCount.a429Prog;
                }

                /* If no fixed rx channels, check for existance of programmable */
                if (u8NumRx == 0)
                {
                    MessageBox.Show("ARINC 429 card has no fixed rx channels!");

                    if (sChanCount.a429Prog > 0)
                        u8NumRx = sChanCount.a429Prog;
                }

                /* Populate the TxChannel comboBox based on number of tx channels */
                for (u8Chan = 1; u8Chan <= u8NumTx; u8Chan++)
                {
                    comboBox_TxChannel.Items.Add(u8Chan);
                }

                /* Populate the RxChannel comboBox based on number of rx channels */
                for (u8Chan = 1; u8Chan <= u8NumRx; u8Chan++)
                {
                    comboBox_RxChannel.Items.Add(u8Chan);
                }

                if (u8NumTx > 0)
                {
                    comboBox_TxChannel.SelectedIndex = 0;
                    button_EnableTx.Enabled = true;
                }

                if (u8NumRx > 0)
                {
                    comboBox_RxChannel.SelectedIndex = 0;
                    button_EnableRx.Enabled = true;
                }

                s16Result = InstallFifoRxHostBuffer(s16Card, 0);
                if (s16Result != ERR_SUCCESS)
                {
                    MessageBox.Show("InstallFifoRxHostBuffer Failed", "ARINC 429 Card Initialisation", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
                else
                {
                    // MessageBox.Show("InstallFifoRxHostBuffer success!");
                }

                MessageBox.Show("ARINC 429 card initialisation completed successfully", "ARINC 429 Card Initialisation", MessageBoxButtons.OK, MessageBoxIcon.Information );
                button_InitCard.Enabled = false;
                button_FreeCard.Enabled = true;
            }
        }

        private void button_EnableTx_Click(object sender, EventArgs e)
        {
            /* Initialize transmitters on the selected Tx channel */
            if (comboBox_TxChannel.SelectedItem != null)
            {
                u8Chan = Byte.Parse(comboBox_TxChannel.SelectedItem.ToString());

                s16Card = Int16.Parse(comboBox_CardNum.SelectedItem.ToString());

                s16Result = EnableTx(s16Card, u8Chan, DD429_ENABLE);
                s16Result += SetTxSpeed(s16Card, u8Chan, (S16BIT) Speed.DD429_LOW_SPEED);

                if (s16Result != ERR_SUCCESS)
                {
                    MessageBox.Show("Enable Tx Failed", "ARINC 429 Tx Channel Enable", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                else
                {
                    MessageBox.Show("Enable Tx success!", "ARINC 429 Tx Channel Enable", MessageBoxButtons.OK, MessageBoxIcon.Information);
                    button_EnableTx.Enabled = false;
                    button_DisableTx.Enabled = true;
                    button_Tx.Enabled = true;
                    textBoxHexData.Enabled = true;
                }
            }
            else
            {
                MessageBox.Show("Tx Channel not selected!", "ARINC 429 Tx Channel Enable", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        private void button_Tx_Click(object sender, EventArgs e)
        {
            if (comboBox_TxChannel.SelectedItem != null)
            {
                u8Chan = Byte.Parse(comboBox_TxChannel.SelectedItem.ToString());

                s16Card = Int16.Parse(comboBox_CardNum.SelectedItem.ToString());

                uint num;
                try
                {
                    num = uint.Parse(textBoxHexData.Text, System.Globalization.NumberStyles.AllowHexSpecifier);
                    s16Result = LoadTxQueueOne(s16Card, u8Chan, num);
                    if (s16Result != 1)
                    {
                        MessageBox.Show("LoadTxQueueOne Failed", "ARINC 429 Transmit Message", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                    else
                    {
                        MessageBox.Show("Transmit success!", "ARINC 429 Transmit Message", MessageBoxButtons.OK, MessageBoxIcon.Information);
                    }
                }
                catch
                {
                    MessageBox.Show("Invalid data value entered. Please enter a valid hex value.", "ARINC 429 Transmit Message", MessageBoxButtons.OK, MessageBoxIcon.Stop);

                    textBoxHexData.Focus();
                    textBoxHexData.SelectAll();
                }
            }
            else
            {
                MessageBox.Show("Tx Channel not selected!", "ARINC 429 Transmit Message", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        private void button_DisableTx_Click(object sender, EventArgs e)
        {
            /* Disable transmitters on the selected Tx channel */
            if (comboBox_TxChannel.SelectedItem != null)
            {
                u8Chan = Byte.Parse(comboBox_TxChannel.SelectedItem.ToString());

                s16Card = Int16.Parse(comboBox_CardNum.SelectedItem.ToString());

                s16Result = EnableTx(s16Card, u8Chan, DD429_DISABLE);

                if (s16Result != ERR_SUCCESS)
                {
                    MessageBox.Show("Disable Tx Failed", "ARINC 429 Tx Channel Disable", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                else
                {
                    MessageBox.Show("Disable Tx success!", "ARINC 429 Tx Channel Disable", MessageBoxButtons.OK, MessageBoxIcon.Information);
                    button_EnableTx.Enabled = true;
                    button_DisableTx.Enabled = false;
                    button_Tx.Enabled = false;
                    textBoxHexData.Enabled = false;
                }
            }
            else
            {
                MessageBox.Show("Tx Channel not selected!", "ARINC 429 Tx Channel Disable", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        private void button_EnableRx_Click(object sender, EventArgs e)
        {
            /* Initialize the selected Rx channel */
            if (comboBox_RxChannel.SelectedItem != null)
            {
                u8Chan = Byte.Parse(comboBox_RxChannel.SelectedItem.ToString());

                s16Card = Int16.Parse(comboBox_CardNum.SelectedItem.ToString());

                s16Result = SetRxChannelSpeed(s16Card, u8Chan, (S16BIT) Speed.DD429_LOW_SPEED);
                s16Result += SetRxChannelParity(s16Card, u8Chan, (S16BIT) Parity.DD429_NO_PARITY);
                s16Result += SetRxChannelMode(s16Card, u8Chan, FIFO_MODE);
                s16Result += EnableRxHostBuffer(s16Card, u8Chan, 0);
                s16Result += EnableTimeStamp(s16Card, u8Chan, DD429_ENABLE);
                s16Result += EnableFilter(s16Card, u8Chan, DD429_DISABLE);
                s16Result += EnableRx(s16Card, u8Chan, DD429_ENABLE);

                if (s16Result != ERR_SUCCESS)
                {
                    MessageBox.Show("Enable Rx Failed", "ARINC 429 Rx Channel Enable", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                else
                {
                    MessageBox.Show("Enable Rx success!", "ARINC 429 Rx Channel Enable", MessageBoxButtons.OK, MessageBoxIcon.Information);
                    button_EnableRx.Enabled = false;
                    button_DisableRx.Enabled = true;
                    button_Rx1Msg.Enabled = true;
                    button_RxAllMsgs.Enabled = true;
                }
            }
            else
            {
                MessageBox.Show("Rx Channel not selected!", "ARINC 429 Rx Channel Enable", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        private void button_ClearDisp_Click(object sender, EventArgs e)
        {
            richTextBox_RxdMsgs.Clear();
        }

        private void button_DisableRx_Click(object sender, EventArgs e)
        {
            /* Disable the selected Rx channel */
            if (comboBox_RxChannel.SelectedItem != null)
            {
                u8Chan = Byte.Parse(comboBox_RxChannel.SelectedItem.ToString());

                s16Card = Int16.Parse(comboBox_CardNum.SelectedItem.ToString());

                s16Result = DisableRxHostBuffer(s16Card, u8Chan);
                s16Result += ResetRxChannel(s16Card, u8Chan);

                if (s16Result != ERR_SUCCESS)
                {
                    MessageBox.Show("Disable Rx Failed", "ARINC 429 Rx Channel Disable", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                else
                {
                    MessageBox.Show("Disable Rx success!", "ARINC 429 Rx Channel Disable", MessageBoxButtons.OK, MessageBoxIcon.Information);
                    button_EnableRx.Enabled = true;
                    button_DisableRx.Enabled = false;
                    button_Rx1Msg.Enabled = false;
                    button_RxAllMsgs.Enabled = false;
                }
            }
            else
            {
                MessageBox.Show("Rx Channel not selected!", "ARINC 429 Rx Channel Disable", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        private void button_Rx1Msg_Click(object sender, EventArgs e)
        {
            if (comboBox_RxChannel.SelectedItem != null)
            {
                U16BIT u16NumMessagesRead = 0;

                u8Chan = Byte.Parse(comboBox_RxChannel.SelectedItem.ToString());

                s16Card = Int16.Parse(comboBox_CardNum.SelectedItem.ToString());

                s16Result = ReadRxHostBuffer(s16Card, u8Chan, 1, ref u16NumMessagesRead, sRxHbufMessages);

                if (s16Result == ERR_OVERFLOW)
                {
                    MessageBox.Show("Overflow error", "ARINC 429 Receive Message", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                else if (s16Result == ERR_RX_HBUF_OVERFLOW)
                {
                    MessageBox.Show("host buffer overflow error", "ARINC 429 Receive Message", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                else if (s16Result == ERR_SUCCESS)
                {
                    if(u16NumMessagesRead==1)
                    {
                        richTextBox_RxdMsgs.AppendText("Data = 0x" + sRxHbufMessages[0].u32Data.ToString("X8") +
                                                        ", TT High = 0x" + sRxHbufMessages[0].u32StampHigh.ToString("X8") +
                                                        ", TT Low = 0x" + sRxHbufMessages[0].u32StampLow.ToString("X8") + Environment.NewLine);
                    }
                    else
                    {
                        MessageBox.Show("No message to read!", "ARINC 429 Receive Message", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                    }
                }
                else
                {
                    MessageBox.Show("Receive error! Have you enabled the Rx channel?", "ARINC 429 Receive Message", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            else
            {
                MessageBox.Show("Rx Channel not selected!", "ARINC 429 Receive Message", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        private void button_RxAllMsgs_Click(object sender, EventArgs e)
        {
            if (comboBox_RxChannel.SelectedItem != null)
            {
                U16BIT u16NumMessagesRead = 0;

                u8Chan = Byte.Parse(comboBox_RxChannel.SelectedItem.ToString());

                s16Card = Int16.Parse(comboBox_CardNum.SelectedItem.ToString());

                s16Result = ReadRxHostBuffer(s16Card, u8Chan, MAX_NUM_MESSAGES_TO_READ, ref u16NumMessagesRead, sRxHbufMessages);

                if (s16Result == ERR_OVERFLOW)
                {
                    MessageBox.Show("Overflow error", "ARINC 429 Receive Message", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                else if (s16Result == ERR_RX_HBUF_OVERFLOW)
                {
                    MessageBox.Show("host buffer overflow error", "ARINC 429 Receive Message", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
                else if (s16Result == ERR_SUCCESS)
                {
                    if (u16NumMessagesRead > 0)
                    {
                        for (int ctr = 0; ctr < u16NumMessagesRead; ctr++)
                        {
                            richTextBox_RxdMsgs.AppendText("Data = 0x" + sRxHbufMessages[ctr].u32Data.ToString("X8") +
                                                        ", TT High = 0x" + sRxHbufMessages[ctr].u32StampHigh.ToString("X8") +
                                                        ", TT Low = 0x" + sRxHbufMessages[ctr].u32StampLow.ToString("X8") + Environment.NewLine);
                        }
                    }
                    else
                    {
                        MessageBox.Show("No message to read!", "ARINC 429 Receive Message", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                    }
                }
                else
                {
                    MessageBox.Show("Receive error! Have you enabled the Rx channel?", "ARINC 429 Receive Message", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            else
            {
                MessageBox.Show("Rx Channel not selected!", "ARINC 429 Receive Message", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        public arinc429_fifo_form()
        {
            InitializeComponent();

            comboBox_CardNum.SelectedIndex = 0;
        }
    }
}
