/******************************************************************************
 *
 * RTips Technologies, Bangalore (www.rtipsonline.com)
 *
 * This sample demonstrates using DDC's AceXtreme SDK in a Qt GUI application.
 *
 * Many portions of this file are adapted from the bcdemo.c sample provided in
 * the AceXtreme SDK (BU-69092Sx) of M/s. Data Device Corporation
 *
 * 1. This sample demonstrates using DDC's AceXtreme SDK in a Qt GUI application.
 * The sample creates two BC->RT messages and two RT to BC messages. It reads BC host buffer
 * and displays the messages in text edits. It allows the user to choose messages to be
 * transmitted on Bus A or B and also optionally enable message retry. If enabled, retry
 * happens once on the alternate bus. Lastly, it demonstrates the aceBCDataBlkWrite API to
 * update data for BC->RT messages.
 *	API's demonstrated in this sample
 *	aceInitialize
 *	aceBCDataBlkCreate
 *	aceBCSetMsgRetry
 *	aceBCMsgCreateBCtoRT
 *	aceBCMsgCreateRTtoBC
 *	aceBCOpCodeCreate
 *	aceBCFrameCreate
 *	aceBCInstallHBuf
 *	aceBCStart
 *	aceBCGetHBufMsgDecoded
 *	aceCmdWordParse
 *	aceBCStop
 *	aceFree
 *	aceBCDataBlkWrite
 *
 * Author: Ganesh Okade, Subhashini BS
 * Last updated: Dec 03, 2024
 ******************************************************************************/
#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QCloseEvent>
#include <QMessageBox>

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
    , ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    /* Connect the signal emitted by msgGetThread to a slot */
    QObject::connect(&m_msgGetThread,
                     SIGNAL(NewHbufMsgAvlbl(int,MSGSTRUCT)),
                     this,
                     SLOT(ProcessNewHbufMsgAvlbl(int,MSGSTRUCT)));

    ui->pushButton_FreeChannel->setEnabled(false);
    ui->pushButton_InitBC->setEnabled(true);
    ui->pushButton_StopBC->setEnabled(false);
    ui->pushButton_StartBC->setEnabled(false);
    ui->pushButton_Write->setEnabled(false);
    ui->radioButton_BusA->setEnabled(true);
    ui->radioButton_BusB->setEnabled(true);
    ui->checkBox_Retry->setEnabled(true);
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    // Check if BC is running. If yes, abort with a message
    if(!ui->pushButton_InitBC->isEnabled())
    {
        QMessageBox::critical(this, "Close", "Cannot close application while channel is in use. Free the channel and retry.");
        event->ignore();
    }
}


void MainWindow::on_pushButton_InitBC_clicked()
{
    int wResult;
    S16BIT aOpCodes[10] = { 0x0000 };
    U32BIT u32MsgOptions = 1;

    /* Buffer containing data for initial values of data blocks */
    U16BIT wBuffer[32] =
    {
        0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
        0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
        0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
        0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004
    };

    /* Get user selected 1553 channel number (Logical Device Number) */
    m_devNum = ui->comboBox_LDN->currentIndex();

    /* Initialize the channel as a BC */
    wResult = aceInitialize(m_devNum, ACE_ACCESS_CARD, ACE_MODE_BC, 0, 0, 0);
    if (wResult)
    {
        QMessageBox::critical(this, "Init BC", "aceInitialize Failed");
        return;
    }

    /* Create data blocks */
    aceBCDataBlkCreate(m_devNum, DBLK1, 32, wBuffer, 32);
    aceBCDataBlkCreate(m_devNum, DBLK2, 32, wBuffer, 32);

    /* Different data for the second BC->RT message */
    for (int i = 0; i < 32; i++)
    {
        wBuffer[i] = i;
    }

    aceBCDataBlkCreate(m_devNum, DBLK3, 32, wBuffer, 32);
    aceBCDataBlkCreate(m_devNum, DBLK4, 32, wBuffer, 32);

    /* Set Bus (A or B) on which message will be transmitted */
    if (ui->radioButton_BusA->isChecked())
        u32MsgOptions = ACE_BCCTRL_CHL_A;
    else
        u32MsgOptions = ACE_BCCTRL_CHL_B;

    /* Set message retry if enabled and also set retry options */
    if (ui->checkBox_Retry->isChecked())
    {
        u32MsgOptions |= ACE_BCCTRL_RETRY_ENA;

        /* Set retry options */
        aceBCSetMsgRetry(m_devNum, ACE_RETRY_ONCE, ACE_RETRY_ALT, ACE_RETRY_ALT, 0);
    }

    /* Create message block */
    aceBCMsgCreateBCtoRT(
        m_devNum,               /* Device number                    */
        MSG1,                   /* Message ID to create             */
        DBLK1,                  /* Message will use this data block */
        m_MsgPara[0].rtAddr,	/* RT address                       */
        m_MsgPara[0].subAddr,   /* RT subaddress                    */
        m_MsgPara[0].wordCnt,   /* Word count                       */
        0,                      /* Default message timer            */
        u32MsgOptions);         /* use chl A options                */

    /* Create XEQ opcode that will use msg block */
    aceBCOpCodeCreate(m_devNum, OP1, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, MSG1, 0, 0);

    aceBCMsgCreateRTtoBC(
        m_devNum,               /* Device number                    */
        MSG2,                   /* Message ID to create             */
        DBLK2,                  /* Message will use this data block */
        m_MsgPara[1].rtAddr,	/* RT address                       */
        m_MsgPara[1].subAddr,   /* RT subaddress                    */
        m_MsgPara[1].wordCnt,   /* Word count                       */
        0,                      /* Default message timer            */
        u32MsgOptions);         /* use chl A options                */

    /* Create XEQ opcode that will use msg block */
    aceBCOpCodeCreate(m_devNum, OP2, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, MSG2, 0, 0);

    aceBCMsgCreateBCtoRT(
        m_devNum,               /* Device number                    */
        MSG3,                   /* Message ID to create             */
        DBLK3,                  /* Message will use this data block */
        m_MsgPara[2].rtAddr,	/* RT address                       */
        m_MsgPara[2].subAddr,   /* RT subaddress                    */
        m_MsgPara[2].wordCnt,   /* Word count                       */
        0,                      /* Default message timer            */
        u32MsgOptions);         /* use chl A options                */

    /* Create XEQ opcode that will use msg block */
    aceBCOpCodeCreate(m_devNum, OP3, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, MSG3, 0, 0);

    aceBCMsgCreateRTtoBC(
        m_devNum,               /* Device number                    */
        MSG4,                   /* Message ID to create             */
        DBLK4,                  /* Message will use this data block */
        m_MsgPara[3].rtAddr,	/* RT address                       */
        m_MsgPara[3].subAddr,   /* RT subaddress                    */
        m_MsgPara[3].wordCnt,   /* Word count                       */
        0,                      /* Default message timer            */
        u32MsgOptions);         /* use chl A options                */

    /* Create XEQ opcode that will use msg block */
    aceBCOpCodeCreate(m_devNum, OP4, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, MSG4, 0, 0);

    /* Create CAL opcode that will call mnr frame from major */
    aceBCOpCodeCreate(m_devNum, OPCAL1, ACE_OPCODE_CAL, ACE_CNDTST_ALWAYS, MNR1, 0, 0);
    aceBCOpCodeCreate(m_devNum, OPCAL2, ACE_OPCODE_CAL, ACE_CNDTST_ALWAYS, MNR2, 0, 0);

    /* Create Minor Frame 1 */
    aOpCodes[0] = OP1;
    aOpCodes[1] = OP2;
    aceBCFrameCreate(m_devNum, MNR1, ACE_FRAME_MINOR, aOpCodes, 2, 0, 0);

    /* Create Minor Frame 2 */
    aOpCodes[0] = OP3;
    aOpCodes[1] = OP4;
    aceBCFrameCreate(m_devNum, MNR2, ACE_FRAME_MINOR, aOpCodes, 2, 0, 0);

    /* Create Major Frame */
    aOpCodes[0] = OPCAL1;
    aOpCodes[1] = OPCAL2;
    aceBCFrameCreate(m_devNum, MJR, ACE_FRAME_MAJOR, aOpCodes, 2, 1000, 0);

    /* Create Host Buffer */
    aceBCInstallHBuf(m_devNum, 32 * 1024);

    ui->pushButton_StartBC->setEnabled(true);
    ui->pushButton_FreeChannel->setEnabled(true);
    ui->pushButton_InitBC->setEnabled(false);
    ui->pushButton_StopBC->setEnabled(false);
    ui->pushButton_Write->setEnabled(false);

    ui->radioButton_BusA->setEnabled(false);
    ui->radioButton_BusB->setEnabled(false);
    ui->checkBox_Retry->setEnabled(false);
}

/* This slot processes the received message for user-friendly display
 * in the respective text edit control
 */
void MainWindow::ProcessNewHbufMsgAvlbl(int msgNum, MSGSTRUCT strMsg)
{
    QTextEdit* textEditMsg[] = {ui->textEdit_Msg1, ui->textEdit_Msg2, ui->textEdit_Msg3, ui->textEdit_Msg4};
    QString displayStr;

    /* Update display with data from the received message */
    textEditMsg[msgNum]->clear();
    displayStr = CreateDisplayString(&strMsg);
    textEditMsg[msgNum]->moveCursor(QTextCursor::End);
    textEditMsg[msgNum]->insertPlainText(displayStr);
}

QString MainWindow::CreateDisplayString(MSGSTRUCT *pMsg)
{
    U16BIT i;
    char szBuffer[100];
    U16BIT wRT, wTR1, wTR2, wSA, wWC;
    QString tmpStr, retStr;

    /* Display message header info */
    tmpStr = "TIME = " + QString::number(pMsg->wTimeTag * 2).rightJustified(8, '0') + "us    BUS " + (pMsg->wBlkSts & ACE_BC_BSW_CHNL ? "B" : "A") +
            "   TYPE" + QString::number(pMsg->wType) + ": " + QString::fromUtf8(aceGetMsgTypeString(pMsg->wType)) + " ";
    retStr += tmpStr;

    /* Display command word info */
    aceCmdWordParse(pMsg->wCmdWrd1, &wRT, &wTR1, &wSA, &wWC);
    sprintf_s(szBuffer, "%02d-%c-%02d-%02d", wRT, wTR1 ? 'T' : 'R', wSA, wWC);
    tmpStr = "\n            CMD1 " + QString::number(pMsg->wCmdWrd1, 16).rightJustified(4, '0').toUpper() + " --> " + QString::fromUtf8(szBuffer);
    retStr += tmpStr;

    tmpStr = "\tBSW " + QString::number(pMsg->wBlkSts, 16).rightJustified(4, '0').toUpper() + " ";
    retStr += tmpStr;

    if (pMsg->wCmdWrd2Flg)
    {
        aceCmdWordParse(pMsg->wCmdWrd2, &wRT, &wTR2, &wSA, &wWC);
        sprintf_s(szBuffer, "%02d-%c-%02d-%02d", wRT, wTR2 ? 'T' : 'R', wSA, wWC);
        tmpStr = "\n            CMD2 " + QString::number(pMsg->wCmdWrd2, 16).rightJustified(4, '0').toUpper() + " --> " + QString::fromUtf8(szBuffer);
        retStr += tmpStr;
    }

    /* Display transmit status words */
    if ((wTR1 == 1) || (pMsg->wBCCtrlWrd & ACE_BCCTRL_RT_TO_RT))
    {
        if (pMsg->wStsWrd1Flg)
        {
            tmpStr = "\n            STA1 " + QString::number(pMsg->wStsWrd1, 16).rightJustified(4, '0').toUpper();
            retStr += tmpStr;
        }
    }

    /* Display Data words */
    for (i = 0; i < pMsg->wWordCount; i++)
    {
        if (i == 0)
        {
            tmpStr = "\n            DATA ";
            retStr += tmpStr;
        }

        tmpStr = QString::number(pMsg->aDataWrds[i], 16).rightJustified(4, '0').toUpper() + "  ";
        retStr += tmpStr;

        if ((i % 8) == 7)
        {
            tmpStr = "\n                 ";
            retStr += tmpStr;
        }
    }

    /* Display receive status words */
    if ((wTR1 == 0) && !(pMsg->wBCCtrlWrd & ACE_BCCTRL_RT_TO_RT))
    {
        if (pMsg->wStsWrd1Flg)
        {
            tmpStr = "\n            STA1 " + QString::number(pMsg->wStsWrd1, 16).rightJustified(4, '0').toUpper();
            retStr += tmpStr;
        }
    }

    if (pMsg->wStsWrd2Flg)
    {
        tmpStr = "\n            STA2 " + QString::number(pMsg->wStsWrd2, 16).rightJustified(4, '0').toUpper();
        retStr += tmpStr;
    }

    /* Display Error information */
    if (pMsg->wBlkSts & 0x170f)
    {
        tmpStr = "\n ERROR: " + QString::fromUtf8(aceGetBSWErrString(ACE_MODE_BC, pMsg->wBlkSts));
        retStr += tmpStr;
    }

    return retStr;
}

void MainWindow::on_pushButton_StartBC_clicked()
{
    int wResult;
    int wRepeatCount = -1;

    /* Start BC */
    wResult = aceBCStart(m_devNum, MJR, wRepeatCount);
    if (wResult)
    {
        char strBuf[250];
        aceErrorStr(wResult, strBuf, 250);
        QMessageBox::critical(this, "Start BC", "BCStart Failed: " + QString::fromUtf8(strBuf));
        return;
    }

    /* Change state of buttons */
    ui->pushButton_StopBC->setEnabled(true);
    ui->pushButton_Write->setEnabled(true);
    ui->pushButton_StartBC->setEnabled(false);
    ui->pushButton_FreeChannel->setEnabled(false);

    /* Create a thread to read BC host buffer msgs */
    m_msgGetThread.SetPtr(this);
    m_msgGetThread.stop = false;
    m_msgGetThread.start();
}


void MainWindow::on_pushButton_StopBC_clicked()
{
    int wResult;

    /* Stop Thread */
    m_msgGetThread.stop = true;
    /* Wait to allow the thread to exit */
    while(m_msgGetThread.IsThreadStopped()==false);

    /* Stop BC */
    wResult = aceBCStop(m_devNum);
    if (wResult)
    {
        QMessageBox::critical(this, "Stop BC", "BCStop Failed");
        return;
    }

    /* Change state of buttons */
    ui->pushButton_StopBC->setEnabled(false);
    ui->pushButton_StartBC->setEnabled(true);
    ui->pushButton_FreeChannel->setEnabled(true);
    ui->pushButton_Write->setEnabled(false);
}


void MainWindow::on_pushButton_FreeChannel_clicked()
{
    /* Free the card */
    S16BIT wResult = aceFree(m_devNum);
    if (wResult)
    {
        QMessageBox::critical(this, "Free Channel", "aceFree Failed");
        return;
    }

    /* Change state of buttons */
    ui->pushButton_FreeChannel->setEnabled(false);
    ui->pushButton_InitBC->setEnabled(true);
    ui->pushButton_StopBC->setEnabled(false);
    ui->pushButton_StartBC->setEnabled(false);

    ui->radioButton_BusA->setEnabled(true);
    ui->radioButton_BusB->setEnabled(true);
    ui->checkBox_Retry->setEnabled(true);
}


void MainWindow::on_pushButton_Close_clicked()
{
    // Check if BC is running. If yes, abort with a message
    if(!ui->pushButton_InitBC->isEnabled())
    {
        QMessageBox::critical(this, "Close", "Cannot close application while channel is in use. Free the channel and retry.");
    }
    else
    {
        this->close();
    }
}


void MainWindow::on_pushButton_Write_clicked()
{
    U16BIT wBuffer[32];
    S16BIT dblkId = -1;

    // Get current msg selection
    int msgSelected = ui->comboBox_MsgSelect->currentIndex();

    // Choose the correct data block to write based on the message selection
    dblkId = (msgSelected == 0) ? DBLK1 : DBLK3;

    // Validate the data value entered by the user
    if ((ui->lineEdit_DataToWrite->text().toInt() < 0) || ui->lineEdit_DataToWrite->text().toInt() > 0xFFFF) // Valid values are 0 to 65535
    {
        QMessageBox::critical(this, "Write", "Value out of range!");
        return;
    }

    // Populate a 32 word buffer with data input by the user
    for(int i = 0; i < 32; i++)
    {
        wBuffer[i] = (U16BIT)ui->lineEdit_DataToWrite->text().toInt();
    }

    // call aceBCDataBlkWrite to write the data to the data block
    S16BIT wResult = aceBCDataBlkWrite(m_devNum, dblkId, wBuffer, 32, 0);
    if (wResult)
    {
        QMessageBox::critical(this, "Write", "aceBCDataBlkWrite Failed");
        return;
    }
}

