﻿/******************************************************************************
 *
 * RTips Technologies, Bangalore (www.rtipsonline.com)
 *
 * This sample demonstrates using DDC's AceXtreme SDK in a C# Windows Forms
 * GUI application. The sample creates two BC->RT messages.
 *
 * Many portions of this file are adapted from the bcdemo.c sample provided in
 * the AceXtreme SDK (BU-69092Sx) of M/s. Data Device Corporation
 * 
 * 1. This sample demonstrates using DDC's AceXtreme SDK in a C# Windows Forms GUI application. 
 * The sample creates two BC->RT messages and two RT to BC messages. It reads BC host buffer 
 * and displays the messages in rich text boxes. It allows the user to choose messages to be 
 * transmitted on Bus A or B and also optionally enable message retry. If enabled, retry 
 * happens once on the alternate bus. Lastly, it demonstrates the aceBCDataBlkWrite API to 
 * update data for BC->RT messages.
 *	API's demonstrated in this sample 
 *	aceInitialize
 *	aceBCDataBlkCreate
 *	aceBCSetMsgRetry
 *	aceBCMsgCreateBCtoRT
 *	aceBCMsgCreateRTtoBC
 *	aceBCOpCodeCreate
 *	aceBCFrameCreate
 *	aceBCInstallHBuf
 *	aceBCStart
 *	aceBCGetHBufMsgDecoded
 *	aceCmdWordParse
 *	aceBCStop
 *	aceFree
 *	aceBCDataBlkWrite
 *	aceGetMsgTypeString
 *	aceGetBSWErrString
 *	aceErrorStr
 *
 * Author: Subhashini
 * Last updated: Feb 28, 2025
 ******************************************************************************/

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Runtime.InteropServices;
using System.Threading;
using System.Globalization;

namespace bcdemo_WinForms_gui
{
    using U64BIT = System.UInt64;
    using U32BIT = System.UInt32;
    using U16BIT = System.UInt16;
    using U8BIT = System.Byte;
    using S64BIT = System.Int64;
    using S32BIT = System.Int32;
    using S16BIT = System.Int16;
    using S8BIT = System.SByte;

    /* Global (used for all modes) Message Structure for decoded 1553 msgs */
    public struct MSGSTRUCT
    {
        public U16BIT wType;               /* Contains the msg type (see above) */
        public U16BIT wBlkSts;             /* Contains the block status word */
        public U16BIT wTimeTag;            /* Time tag of message */
        public U16BIT wCmdWrd1;            /* First command word */
        public U16BIT wCmdWrd2;            /* Second command word (RT to RT) */
        public U16BIT wCmdWrd1Flg;         /* Is command word 1 valid? */
        public U16BIT wCmdWrd2Flg;         /* Is command word 2 valid? */
        public U16BIT wStsWrd1;            /* First status word */
        public U16BIT wStsWrd2;            /* Second status word */
        public U16BIT wStsWrd1Flg;         /* Is status word 1 valid? */
        public U16BIT wStsWrd2Flg;         /* Is status word 2 valid? */
        public U16BIT wWordCount;          /* Number of valid data words */

        [MarshalAs(UnmanagedType.ByValArray, SizeConst = 32)]
        public U16BIT[] aDataWrds;       /* An array of data words */

        /* The following are only applicable in BC mode */
        public U16BIT wBCCtrlWrd;          /* Contains the BC control word */
        public U16BIT wBCGapTime;          /* Message gap time word */
        public U16BIT wBCLoopBack1;        /* First looped back word */
        public U16BIT wTimeTag2;           /* wBCLoopBack2 is redefined as TimeTag2 */
        public U16BIT wBCLoopBack1Flg;     /* Is loop back 1 valid? */
        public U16BIT wTimeTag3;           /* wBCLoopBack2Flg is redefined as TimeTag3 */
    }

    public struct MSG_PARAMS
    {
        public U8BIT rtAddr;
        public U8BIT TR;
        public U8BIT subAddr;
        public U8BIT wordCnt;
    }

    public partial class bcdemo_Form : Form
    {
        #region Imported API from DDC SDK Dll
            [DllImport("emacepl.dll")]
            static extern S16BIT aceInitialize(
                S16BIT DevNum,
                U16BIT wAccess,
                U16BIT wMode,
                U32BIT dwMemWrdSize,
                U32BIT dwRegAddr,
                U32BIT dwMemAddr
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceFree(
                S16BIT DevNum
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceBCDataBlkCreate(
                S16BIT DevNum,
                S16BIT nDataBlkID,
                U16BIT wDataBlkType,
                U16BIT[] pBuffer,
                U16BIT wBufferSize
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceBCSetMsgRetry(
                S16BIT DevNum,
                U16BIT wNumOfRetries,
                U16BIT wFirstRetryBus,
                U16BIT wSecondRetryBus,
                U16BIT wReserved
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceBCMsgCreateBCtoRT(
                S16BIT DevNum,
                S16BIT nMsgBlkID,
                S16BIT nDataBlkID,
                U16BIT wRT,
                U16BIT wSA,
                U16BIT wWC,
                U16BIT wMsgGapTime,
                U32BIT dwMsgOptions
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceBCOpCodeCreate(
                S16BIT DevNum,
                S16BIT nOpCodeID,
                U16BIT wOpCodeType,
                U16BIT wCondition,
                U32BIT dwParameter1,
                U32BIT dwParameter2,
                U32BIT dwReserved
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceBCMsgCreateRTtoBC(
                S16BIT DevNum,
                S16BIT nMsgBlkID,
                S16BIT nDataBlkID,
                U16BIT wRT,
                U16BIT wSA,
                U16BIT wWC,
                U16BIT wMsgGapTime,
                U32BIT dwMsgOptions
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceBCFrameCreate(
                S16BIT DevNum,
                S16BIT nFrameID,
                U16BIT wFrameType,
                S16BIT[] aOpCodeIDs,
                U16BIT wOpCodeCount,
                U16BIT wMnrFrmTime,
                U16BIT wFlags
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceBCInstallHBuf(
                S16BIT DevNum,
                U32BIT dwHBufSize
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceCmdWordParse(
                U16BIT wCmdWrd,
                out U16BIT pRT,
                out U16BIT pTR,
                out U16BIT pSA,
                out U16BIT pWC
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceBCStart(
                S16BIT DevNum,
                S16BIT nMjrFrmID,
                S32BIT lMjrFrmCount
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceBCGetHBufMsgDecoded(
                S16BIT DevNum,
                out MSGSTRUCT pMsg,
                out U32BIT pdwMsgCount,
                out U32BIT pdwMsgLostHBuf,
                U16BIT wMsgLoc
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceBCStop(
                S16BIT DevNum
            );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceBCDataBlkWrite(
                S16BIT DevNum,
                S16BIT nDataBlkID,
                U16BIT[] pBuffer,
                U16BIT wBufferSize,
                U16BIT wOffset
            );

            [DllImport("emacepl.dll")]
            static extern IntPtr aceGetMsgTypeString(
                    U16BIT wMsgType
                );

            [DllImport("emacepl.dll")]
            static extern IntPtr aceGetBSWErrString(
                    U16BIT wMode,
                    U16BIT wBlkSts
                );

            [DllImport("emacepl.dll")]
            static extern S16BIT aceErrorStr(
                    S16BIT nError,
                    out string pBuffer,
                    U16BIT wBufSize
                );
        #endregion

        public const int NO_OF_MSGS = 4;

        /* define data blocks */
        public const S16BIT DBLK1 = 1;
        public const S16BIT DBLK2 = 2;
        public const S16BIT DBLK3 = 3;
        public const S16BIT DBLK4 = 4;

        /* define message constants */
        public const S16BIT MSG1 = 1;
        public const S16BIT MSG2 = 2;
        public const S16BIT MSG3 = 3;
        public const S16BIT MSG4 = 4;

        /* define opcodes */
        public const S16BIT OP1 = 1;
        public const S16BIT OP2 = 2;
        public const S16BIT OP3 = 3;
        public const S16BIT OP4 = 4;

        public const S16BIT OPCAL1 = 100;
        public const S16BIT OPCAL2 = 101;

        /* define frame constants */
        public const S16BIT MNR1 = 1;
        public const S16BIT MNR2 = 2;
        public const S16BIT MJR = 100;

        /* define DDC SDK constants */
        public const U16BIT ACE_ACCESS_CARD = 0;
        public const U16BIT ACE_MODE_BC = 0x0001;
        public const U16BIT ACE_BCCTRL_CHL_A = 0x0080;
        public const U16BIT ACE_BCCTRL_CHL_B = 0x0000;
        public const U16BIT ACE_BCCTRL_RETRY_ENA = 0x0100;
        public const U16BIT ACE_RETRY_ONCE = 1;
        public const U16BIT ACE_RETRY_ALT = 1;
        public const U16BIT ACE_OPCODE_XEQ = 0x0001;
        public const U16BIT ACE_CNDTST_ALWAYS = 0x000F;
        public const U16BIT ACE_OPCODE_CAL = 0x0003;
        public const U16BIT ACE_FRAME_MINOR = 0x0002;
        public const U16BIT ACE_FRAME_MAJOR = 0x0000;
        public const U16BIT ACE_BC_BSW_CHNL = 0x2000;
        public const U16BIT ACE_BCCTRL_RT_TO_RT = 0x0001;
        public const U16BIT ACE_BC_MSGLOC_NEXT_PURGE = 0;

        public MSG_PARAMS[] gMsgPara = new MSG_PARAMS[4];

        public S16BIT m_devNum;
        public Thread threadMsgGet;

        /// Delegate function to use for message update in rich text boxes
        public delegate void MessageUpdateDelegate(int msgNum, MSGSTRUCT strMsg);

        /// The delegated method for updating message
        public MessageUpdateDelegate mMessageDelegate;

        public bcdemo_Form()
        {
            InitializeComponent();

            gMsgPara[0].rtAddr = 1;
            gMsgPara[0].TR = 0;
            gMsgPara[0].subAddr = 1;
            gMsgPara[0].wordCnt = 0;

            gMsgPara[1].rtAddr = 1;
            gMsgPara[1].TR = 1;
            gMsgPara[1].subAddr = 2;
            gMsgPara[1].wordCnt = 0;

            gMsgPara[2].rtAddr = 3;
            gMsgPara[2].TR = 0;
            gMsgPara[2].subAddr = 3;
            gMsgPara[2].wordCnt = 0;

            gMsgPara[3].rtAddr = 3;
            gMsgPara[3].TR = 1;
            gMsgPara[3].subAddr = 4;
            gMsgPara[3].wordCnt = 0;

            radioButtonBusA.Checked = true;
            radioButtonBusB.Checked = false;

            buttonFreeBC.Enabled = false;
            buttonInitBC.Enabled = true;
            button_StopBC.Enabled = false;
            button_StartBC.Enabled = false;
            button_Write.Enabled = false;
            radioButtonBusA.Enabled = true;
            radioButtonBusB.Enabled = true;
            checkBox_Retry.Enabled = true;

            comboBox_LDN.SelectedIndex = 0;
            comboBox_MsgWrite.SelectedIndex = 0;

            //Create the delegate for the message update.
            mMessageDelegate = new MessageUpdateDelegate(UpdateMessage);
        }

        /*******************************************************************************
         * Name:    UpdateMessage
         *
         * Description:
         *
         *      This is a delegate handler for the message posted by ThreadMsgGet
         * when it finds a message in the BC Host Buffer.
         * 
         * The handler processes the received message for user-friendly display 
         * in the respective rich text box 
         *
         ******************************************************************************/
        private void UpdateMessage(int msgNum, MSGSTRUCT strMsg)
        {
            string displayStr;
            RichTextBox[] textBoxMsg = new RichTextBox[4]
            {
                richTextBox_Msg1, richTextBox_Msg2, richTextBox_Msg3, richTextBox_Msg4
            };

            displayStr = CreateDisplayString(strMsg);

            /* Update display with data from the received message */
            textBoxMsg[msgNum].Clear();
            textBoxMsg[msgNum].AppendText(displayStr + Environment.NewLine);
        }

        private void buttonInitBC_Click(object sender, EventArgs e)
        {
            S16BIT wResult;
            S16BIT[] aOpCodes = new S16BIT[10];
            U32BIT u32MsgOptions = 1;

            /* Buffer containing data for initial values of data blocks */
            U16BIT[] wBuffer = new U16BIT[32]
            {
                0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
                0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
                0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
                0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004
            };

            /* Get user selected 1553 channel number (Logical Device Number) */
            m_devNum = (S16BIT) comboBox_LDN.SelectedIndex;

            /* Initialize the channel as a BC */
            wResult = aceInitialize(m_devNum, ACE_ACCESS_CARD, ACE_MODE_BC, 0, 0, 0);
            if(wResult != 0)
            {
                MessageBox.Show("aceInitialize Failed");
                return;
            }

            /* Create data blocks */
            aceBCDataBlkCreate(m_devNum, DBLK1, 32, wBuffer, 32);
            aceBCDataBlkCreate(m_devNum, DBLK2, 32, wBuffer, 32);

            /* Different data for the second BC->RT message */
            for (U16BIT i = 0; i < 32; i++)
            {
                wBuffer[i] = i;
            }

            aceBCDataBlkCreate(m_devNum, DBLK3, 32, wBuffer, 32);
            aceBCDataBlkCreate(m_devNum, DBLK4, 32, wBuffer, 32);

            /* Set Bus (A or B) on which message will be transmitted */
            if (radioButtonBusA.Checked)
                u32MsgOptions = ACE_BCCTRL_CHL_A;
            else
                u32MsgOptions = ACE_BCCTRL_CHL_B;

            /* Set message retry if enabled and also set retry options */
            if (checkBox_Retry.Checked)
            {
                u32MsgOptions |= ACE_BCCTRL_RETRY_ENA;

                /* Set retry options */
                aceBCSetMsgRetry(m_devNum, ACE_RETRY_ONCE, ACE_RETRY_ALT, ACE_RETRY_ALT, 0);
            }

            /* Create message block */
            aceBCMsgCreateBCtoRT(
                m_devNum,              /* Device number                    */
                MSG1,                /* Message ID to create             */
                DBLK1,               /* Message will use this data block */
                gMsgPara[0].rtAddr,  /* RT address                       */
                gMsgPara[0].subAddr, /* RT subaddress                    */
                gMsgPara[0].wordCnt, /* Word count                       */
                0,                   /* Default message timer            */
                u32MsgOptions);   /* use chl A options                */

            /* Create XEQ opcode that will use msg block */
            aceBCOpCodeCreate(m_devNum, OP1, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, (U32BIT) MSG1, 0, 0);

            aceBCMsgCreateRTtoBC(
                m_devNum,              /* Device number                    */
                MSG2,                /* Message ID to create             */
                DBLK2,               /* Message will use this data block */
                gMsgPara[1].rtAddr,  /* RT address                       */
                gMsgPara[1].subAddr, /* RT subaddress                    */
                gMsgPara[1].wordCnt, /* Word count                       */
                0,                   /* Default message timer            */
                u32MsgOptions);   /* use chl A options                */

            /* Create XEQ opcode that will use msg block */
            aceBCOpCodeCreate(m_devNum, OP2, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, (U32BIT) MSG2, 0, 0);

            aceBCMsgCreateBCtoRT(
                m_devNum,              /* Device number                    */
                MSG3,                /* Message ID to create             */
                DBLK3,               /* Message will use this data block */
                gMsgPara[2].rtAddr,  /* RT address                       */
                gMsgPara[2].subAddr, /* RT subaddress                    */
                gMsgPara[2].wordCnt, /* Word count                       */
                0,                   /* Default message timer            */
                u32MsgOptions);   /* use chl A options                */

            /* Create XEQ opcode that will use msg block */
            aceBCOpCodeCreate(m_devNum, OP3, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, (U32BIT) MSG3, 0, 0);

            aceBCMsgCreateRTtoBC(
                m_devNum,              /* Device number                    */
                MSG4,                /* Message ID to create             */
                DBLK4,               /* Message will use this data block */
                gMsgPara[3].rtAddr,  /* RT address                       */
                gMsgPara[3].subAddr, /* RT subaddress                    */
                gMsgPara[3].wordCnt, /* Word count                       */
                0,                   /* Default message timer            */
                u32MsgOptions);   /* use chl A options                */

            /* Create XEQ opcode that will use msg block */
            aceBCOpCodeCreate(m_devNum, OP4, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, (U32BIT) MSG4, 0, 0);

            /* Create CAL opcode that will call mnr frame from major */
            aceBCOpCodeCreate(m_devNum, OPCAL1, ACE_OPCODE_CAL, ACE_CNDTST_ALWAYS, (U32BIT) MNR1, 0, 0);
            aceBCOpCodeCreate(m_devNum, OPCAL2, ACE_OPCODE_CAL, ACE_CNDTST_ALWAYS, (U32BIT) MNR2, 0, 0);

            /* Create Minor Frame 1 */
            aOpCodes[0] = OP1;
            aOpCodes[1] = OP2;
            aceBCFrameCreate(m_devNum, MNR1, ACE_FRAME_MINOR, aOpCodes, 2, 0, 0);

            /* Create Minor Frame 2 */
            aOpCodes[0] = OP3;
            aOpCodes[1] = OP4;
            aceBCFrameCreate(m_devNum, MNR2, ACE_FRAME_MINOR, aOpCodes, 2, 0, 0);

            /* Create Major Frame */
            aOpCodes[0] = OPCAL1;
            aOpCodes[1] = OPCAL2;
            aceBCFrameCreate(m_devNum, MJR, ACE_FRAME_MAJOR, aOpCodes, 2, 1000, 0);

            /* Create Host Buffer */
            aceBCInstallHBuf(m_devNum, 32 * 1024);

            button_StartBC.Enabled = true;
            buttonFreeBC.Enabled = true;
            buttonInitBC.Enabled = false;
            button_StopBC.Enabled = false;
            button_Write.Enabled = false;

            radioButtonBusA.Enabled = false;
            radioButtonBusB.Enabled = false;
            checkBox_Retry.Enabled = false;
        }

        string CreateDisplayString(MSGSTRUCT pMsg)
        {
            U16BIT i;
            U16BIT wRT, wTR1, wTR2, wSA, wWC;
            string tmpStr, retStr="";

            /* Display message header info */
            tmpStr = String.Format("TIME = {0:D08}us    BUS {1}   TYPE{2:D}: {3} ", (pMsg.wTimeTag * 2), ((int)(pMsg.wBlkSts & ACE_BC_BSW_CHNL) > 1) ? 'B' : 'A', 
                                        pMsg.wType, Marshal.PtrToStringAnsi(aceGetMsgTypeString(pMsg.wType)));
            retStr += tmpStr;

            /* Display command word info */
            aceCmdWordParse(pMsg.wCmdWrd1, out wRT, out wTR1, out wSA, out wWC);
            tmpStr = String.Format("\n            CMD1 {0:X04} --> {1:D02}-{2}-{3:D02}-{4:D02}", pMsg.wCmdWrd1, wRT, (wTR1 == 1) ? 'T' : 'R', wSA, wWC);
            retStr += tmpStr;

            tmpStr = String.Format("\tBSW {0:X04} ", pMsg.wBlkSts);
            retStr += tmpStr;

            if (pMsg.wCmdWrd2Flg == 1)
            {
                aceCmdWordParse(pMsg.wCmdWrd2, out wRT, out wTR2, out wSA, out wWC);
                tmpStr = String.Format("\n            CMD2 {0:X04} --> {1:D02}-{2}-{3:D02}-{4:D02}", pMsg.wCmdWrd2, wRT, (wTR2 == 1) ? 'T' : 'R', wSA, wWC);
                retStr += tmpStr;
            }

            /* Display transmit status words */
            if ((wTR1 == 1) || ((pMsg.wBCCtrlWrd & ACE_BCCTRL_RT_TO_RT) == ACE_BCCTRL_RT_TO_RT))
            {
                if (pMsg.wStsWrd1Flg == 1)
                {
                    tmpStr = String.Format("\n            STA1 {0:X04}", pMsg.wStsWrd1);
                    retStr += tmpStr;
                }
            }

            /* Display Data words */
            for (i = 0; i < pMsg.wWordCount; i++)
            {
                if (i == 0)
                {
                    tmpStr = "\n            DATA ";
                    retStr += tmpStr;
                }

                tmpStr = String.Format("{0:X04}  ", pMsg.aDataWrds[i]);
                retStr += tmpStr;

                if ((i % 8) == 7)
                {
                    tmpStr = "\n                 ";
                    retStr += tmpStr;
                }
            }

            /* Display receive status words */
            if ((wTR1 == 0) && ((pMsg.wBCCtrlWrd & ACE_BCCTRL_RT_TO_RT) != ACE_BCCTRL_RT_TO_RT))
            {
                if (pMsg.wStsWrd1Flg == 1)
                {
                    tmpStr = String.Format("\n            STA1 {0:X04}", pMsg.wStsWrd1);
                    retStr += tmpStr;
                }
            }

            if (pMsg.wStsWrd2Flg == 1)
            {
                tmpStr = String.Format("\n            STA2 {0:X04}", pMsg.wStsWrd2);
                retStr += tmpStr;
            }

            /* Display Error information */
            if ((pMsg.wBlkSts & 0x170f) != 0)
            {
                tmpStr = String.Format("\n ERROR: {0}", Marshal.PtrToStringAnsi(aceGetBSWErrString((ushort)ACE_MODE_BC, (ushort)pMsg.wBlkSts)));
                retStr += tmpStr;
            }

            return retStr;
        }

        private void buttonFreeBC_Click(object sender, EventArgs e)
        {
            S16BIT wResult;

            /* Free the card */
            wResult = aceFree(m_devNum);
            if (wResult != 0)
            {
                MessageBox.Show("aceFree Failed");
                return;
            }

            /* Change state of buttons */
            buttonFreeBC.Enabled = false;
            buttonInitBC.Enabled = true;
            button_StopBC.Enabled = false;
            button_StartBC.Enabled = false;

            radioButtonBusA.Enabled = true;
            radioButtonBusB.Enabled = true;
            checkBox_Retry.Enabled = true;
        }

        private void button_Close_Click(object sender, EventArgs e)
        {
            // Check if BC is running. If yes, abort with a message
            if (!buttonInitBC.Enabled)
            {
                MessageBox.Show("Cannot close application while channel is in use. Free the channel and retry.");
            }
            else
            {
                this.Close();
            }
        }

        private void button_StartBC_Click(object sender, EventArgs e)
        {
            S16BIT wResult;
            S32BIT wRepeatCount = -1;

            /* Start BC */
            wResult = aceBCStart(m_devNum, MJR, wRepeatCount);
            if (wResult!=0)
            {
                string strBuf;
                aceErrorStr(wResult, out strBuf, 250);
                MessageBox.Show("BCStart Failed: " + strBuf);
                return;
            }

            /* Change state of buttons */
            button_StopBC.Enabled = true;
            button_Write.Enabled = true;
            button_StartBC.Enabled = false;
            buttonFreeBC.Enabled = false;

            /* Create a thread to read BC host buffer msgs */
            threadMsgGet = new Thread(ThreadMsgGet);
            threadMsgGet.Start(m_devNum);
        }

        /*******************************************************************************
         * Name:    ThreadMsgGet
         *
         * Description:
         *
         *      A separate thread to read BC host buffer
         *
         * In   devNum: LDN passed when the thread was started
         * Out  none
         ******************************************************************************/
        void ThreadMsgGet(object devNum)
        {
            S16BIT nResult = 0x0000;
            U32BIT dwMsgCount = 0x00000000;
            U32BIT dwHBufLost = 0x00000000;
            U32BIT dwCurCount = 0x00000000;
            MSGSTRUCT sMsg = new MSGSTRUCT();

            while (true)
            {
                /* Check host buffer for msgs */
                nResult = aceBCGetHBufMsgDecoded((S16BIT)devNum, out sMsg, out dwMsgCount, out dwHBufLost, ACE_BC_MSGLOC_NEXT_PURGE);

                if (nResult!=0)
                {
                    /* Error reading HBuf */
                    continue;
                }

                /* Message found */
                if (dwMsgCount != 0)
                {
                    /* Update total messages counter */
                    ++dwCurCount;

                    /* Analyze the command word to identify this message */
                    U16BIT RT = 0, TR = 0, SA = 0, WC = 0;
                    nResult = aceCmdWordParse(sMsg.wCmdWrd1, out RT, out TR, out SA, out WC);

                    /* Based on the message identified, notify GUI */
                    for (int i = 0; i < NO_OF_MSGS; i++)
                    {
                        if (RT == gMsgPara[i].rtAddr && TR == gMsgPara[i].TR && SA == gMsgPara[i].subAddr && WC == gMsgPara[i].wordCnt) // MSG1
                        {
                            /* Notify GUI thread */
                            Invoke(mMessageDelegate, i, sMsg);
                        }
                    }
                }

                Thread.Sleep(1);
            }
        }

        private void button_StopBC_Click(object sender, EventArgs e)
        {
            S16BIT wResult;

            /* Stop Thread */
            threadMsgGet.Abort();

            /* Wait for a while to allow the thread to exit */
            Thread.Sleep(100);

            /* Stop BC */
            wResult = aceBCStop(m_devNum);
            if (wResult != 0)
            {
                MessageBox.Show("BCStop Failed");
                return;
            }

            /* Change state of buttons */
            button_StopBC.Enabled = false;
            button_Write.Enabled = false;
            button_StartBC.Enabled = true;
            buttonFreeBC.Enabled = true;
        }

        private void button_Write_Click(object sender, EventArgs e)
        {
            U16BIT[] wBuffer = new U16BIT[32];
            S16BIT dblkId = -1;

            // Get current msg selection
            int msgSelected = comboBox_MsgWrite.SelectedIndex;

            // Choose the correct data block to write based on the message selection
            dblkId = (msgSelected == 0) ? DBLK1 : DBLK3;

            try
            {
                // Validate the data value entered by the user
                if ((Int32.Parse(textBox_DataWrite.Text, NumberStyles.HexNumber) < 0) || (Int32.Parse(textBox_DataWrite.Text, NumberStyles.HexNumber) > 0xFFFF)) // Valid values are 0 to 65535
                {
                    MessageBox.Show("Value out of range!");
                    textBox_DataWrite.Focus();
                    return;
                }
            }
            catch (Exception)
            {
                MessageBox.Show("Enter valid value!");
                textBox_DataWrite.Focus();
                return;
            }

            // Populate a 32 word buffer with data input by the user
            for (int i = 0; i < 32; i++)
            {
                wBuffer[i] = UInt16.Parse(textBox_DataWrite.Text, NumberStyles.HexNumber);
            }

            // call aceBCDataBlkWrite to write the data to the data block
            S16BIT wResult = aceBCDataBlkWrite(m_devNum, dblkId, wBuffer, 32, 0);
            if (wResult != 0)
            {
                MessageBox.Show("aceBCDataBlkWrite Failed");
                return;
            }
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            // Check if BC is running. If yes, abort with a message
            if (!buttonInitBC.Enabled)
            {
                MessageBox.Show("Cannot close application while channel is in use. Free the channel and retry.");
                e.Cancel = true;
            }
        }
    }
}
