/******************************************************************************
 *
 * RTips Technologies, Bangalore (www.rtipsonline.com)
 *
 * This sample demonstrates using DDC's AceXtreme SDK in a MS VC++ MFC
 * GUI application. The sample creates two BC->RT messages
 *
 * Many portions of this file are adapted from the bcdemo.c sample provided in
 * the AceXtreme SDK (BU-69092Sx) of M/s. Data Device Corporation
 * 
 * 1. This sample demonstrates using DDC's AceXtreme SDK in a MS VC++ MFC GUI application. 
 * The sample creates two BC->RT messages and two RT to BC messages. It reads BC host buffer 
 * and displays the messages in rich text boxes. It allows the user to choose messages to be 
 * transmitted on Bus A or B and also optionally enable message retry. If enabled, retry 
 * happens once on the alternate bus. Lastly, it demonstrates the aceBCDataBlkWrite API to 
 * update data for BC->RT messages.
 *	API's demonstrated in this sample 
 *	aceInitialize
 *	aceBCDataBlkCreate
 *	aceBCSetMsgRetry
 *	aceBCMsgCreateBCtoRT
 *	aceBCMsgCreateRTtoBC
 *	aceBCOpCodeCreate
 *	aceBCFrameCreate
 *	aceBCInstallHBuf
 *	aceBCStart
 *	aceBCGetHBufMsgDecoded
 *	aceCmdWordParse
 *	aceBCStop
 *	aceFree
 *	aceBCDataBlkWrite
 *
 * Author: Ganesh Okade
 * Last updated: Nov 22, 2024
 ******************************************************************************/

#include "pch.h"
#include "stdemace.h"
#include "framework.h"
#include "bcdemo_mfc_gui.h"
#include "bcdemo_mfc_guiDlg.h"
#include "afxdialogex.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

typedef struct
{
	U8BIT rtAddr;
	U8BIT TR;
	U8BIT subAddr;
	U8BIT wordCnt;
}MSG_PARAMS;

#define NO_OF_MSGS	4

/****** Globals *****/
DWORD         dwIdThreadMsgGet;
HANDLE		  hThreadMsgGet;
BOOL		  quitThread = FALSE;
MSGSTRUCT gMsgs[4];
MSG_PARAMS gMsgPara[4] = { {1, 0, 1, 0},
								{1, 1, 2, 0},
								{3, 0, 3, 0},
								{3, 1, 4, 0}, };

U32BIT dwHBufLost = 0x00000000;
U32BIT dwCurCount = 0x00000000;

#define WM_PROCESS_NEW_HBUF_MSG_AVLBL (WM_USER + 1)

DWORD WINAPI ThreadMsgGet(LPVOID lpParam);

// CAboutDlg dialog used for App About

class CAboutDlg : public CDialogEx
{
public:
	CAboutDlg();

// Dialog Data
#ifdef AFX_DESIGN_TIME
	enum { IDD = IDD_ABOUTBOX };
#endif

	protected:
	virtual void DoDataExchange(CDataExchange* pDX);    // DDX/DDV support

// Implementation
protected:
	DECLARE_MESSAGE_MAP()
};

CAboutDlg::CAboutDlg() : CDialogEx(IDD_ABOUTBOX)
{
}

void CAboutDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialogEx::DoDataExchange(pDX);
}

BEGIN_MESSAGE_MAP(CAboutDlg, CDialogEx)
END_MESSAGE_MAP()


// CbcdemomfcguiDlg dialog



CbcdemomfcguiDlg::CbcdemomfcguiDlg(CWnd* pParent /*=nullptr*/)
	: CDialogEx(IDD_BCDEMO_MFC_GUI_DIALOG, pParent)
	, m_DataToWrite(0)
{
	m_hIcon = AfxGetApp()->LoadIcon(IDR_MAINFRAME);
}

void CbcdemomfcguiDlg::DoDataExchange(CDataExchange* pDX)
{
	CDialogEx::DoDataExchange(pDX);
	DDX_Control(pDX, IDC_COMBO_LDN, m_comboBox);
	DDX_Control(pDX, IDC_BUTTON_START_BC, m_btnStartBC);
	DDX_Control(pDX, IDC_BUTTON_STOP_BC, m_btnStopBC);
	DDX_Control(pDX, IDC_BUTTON_FREE_BC, m_btnFreeBC);
	DDX_Control(pDX, IDC_BUTTON_INIT_BC, m_btnInitializeBC);
	DDX_Text(pDX, IDC_EDIT_DATA_TO_WRITE, m_DataToWrite);
	DDX_Control(pDX, IDC_BUTTON_WRITE, m_btnWrite);
	DDX_Control(pDX, IDC_COMBO__MSG_SELECT, m_comboMsgSelect);
	DDX_Control(pDX, IDC_EDIT_DATA_TO_WRITE, m_editDataToWrite);
	DDX_Control(pDX, IDC_RICHEDIT_MSG1, m_rtbRawMsgDisplay_1);
	DDX_Control(pDX, IDC_RICHEDIT_MSG2, m_rtbRawMsgDisplay_2);
	DDX_Control(pDX, IDC_RICHEDIT_MSG3, m_rtbRawMsgDisplay_3);
	DDX_Control(pDX, IDC_RICHEDIT_MSG4, m_rtbRawMsgDisplay_4);
	DDX_Control(pDX, IDC_RADIO_BUSA, m_rbBusA);
	DDX_Control(pDX, IDC_RADIO_BUSB, m_rbBusB);
	DDX_Control(pDX, IDC_CHECK_RETRY, m_cbMsgRetry);
}

BEGIN_MESSAGE_MAP(CbcdemomfcguiDlg, CDialogEx)
	ON_WM_SYSCOMMAND()
	ON_WM_PAINT()
	ON_WM_QUERYDRAGICON()
	ON_BN_CLICKED(IDC_BUTTON_INIT_BC, &CbcdemomfcguiDlg::OnBnClickedButtonInitBc)
	ON_BN_CLICKED(IDC_BUTTON_START_BC, &CbcdemomfcguiDlg::OnBnClickedButtonStartBc)
	ON_BN_CLICKED(IDC_BUTTON_STOP_BC, &CbcdemomfcguiDlg::OnBnClickedButtonStopBc)
	ON_BN_CLICKED(IDC_BUTTON_FREE_BC, &CbcdemomfcguiDlg::OnBnClickedButtonFreeBc)
	ON_MESSAGE(WM_PROCESS_NEW_HBUF_MSG_AVLBL, &CbcdemomfcguiDlg::OnNewHbufMsgAvailable)
	ON_BN_CLICKED(IDCANCEL, &CbcdemomfcguiDlg::OnBnClickedCancel)
	ON_BN_CLICKED(IDC_BUTTON_WRITE, &CbcdemomfcguiDlg::OnBnClickedButtonWrite)
END_MESSAGE_MAP()


// CbcdemomfcguiDlg message handlers

BOOL CbcdemomfcguiDlg::OnInitDialog()
{
	CDialogEx::OnInitDialog();

	// Add "About..." menu item to system menu.

	// IDM_ABOUTBOX must be in the system command range.
	ASSERT((IDM_ABOUTBOX & 0xFFF0) == IDM_ABOUTBOX);
	ASSERT(IDM_ABOUTBOX < 0xF000);

	CMenu* pSysMenu = GetSystemMenu(FALSE);
	if (pSysMenu != nullptr)
	{
		BOOL bNameValid;
		CString strAboutMenu;
		bNameValid = strAboutMenu.LoadString(IDS_ABOUTBOX);
		ASSERT(bNameValid);
		if (!strAboutMenu.IsEmpty())
		{
			pSysMenu->AppendMenu(MF_SEPARATOR);
			pSysMenu->AppendMenu(MF_STRING, IDM_ABOUTBOX, strAboutMenu);
		}
	}

	// Set the icon for this dialog.  The framework does this automatically
	//  when the application's main window is not a dialog
	SetIcon(m_hIcon, TRUE);			// Set big icon
	SetIcon(m_hIcon, FALSE);		// Set small icon

	// TODO: Add extra initialization here
	m_rbBusA.SetCheck(1);
	m_rbBusB.SetCheck(0);

	m_btnFreeBC.EnableWindow(false);
	m_btnInitializeBC.EnableWindow(true);
	m_btnStopBC.EnableWindow(false);
	m_btnStartBC.EnableWindow(false);
	m_btnWrite.EnableWindow(false);
	m_rbBusA.EnableWindow(true);
	m_rbBusB.EnableWindow(true);
	m_cbMsgRetry.EnableWindow(true);

	m_comboBox.SetCurSel(0);
	m_comboMsgSelect.SetCurSel(0);

	SetWindowText(_T("RTips Technologies - bcdemo GUI sample using MS VC++"));
	return TRUE;  // return TRUE  unless you set the focus to a control
}

void CbcdemomfcguiDlg::OnSysCommand(UINT nID, LPARAM lParam)
{
	if ((nID & 0xFFF0) == IDM_ABOUTBOX)
	{
		CAboutDlg dlgAbout;
		dlgAbout.DoModal();
	}
	else
	{
		CDialogEx::OnSysCommand(nID, lParam);
	}
}

// If you add a minimize button to your dialog, you will need the code below
//  to draw the icon.  For MFC applications using the document/view model,
//  this is automatically done for you by the framework.

void CbcdemomfcguiDlg::OnPaint()
{
	if (IsIconic())
	{
		CPaintDC dc(this); // device context for painting

		SendMessage(WM_ICONERASEBKGND, reinterpret_cast<WPARAM>(dc.GetSafeHdc()), 0);

		// Center icon in client rectangle
		int cxIcon = GetSystemMetrics(SM_CXICON);
		int cyIcon = GetSystemMetrics(SM_CYICON);
		CRect rect;
		GetClientRect(&rect);
		int x = (rect.Width() - cxIcon + 1) / 2;
		int y = (rect.Height() - cyIcon + 1) / 2;

		// Draw the icon
		dc.DrawIcon(x, y, m_hIcon);
	}
	else
	{
		CDialogEx::OnPaint();
	}
}

// The system calls this function to obtain the cursor to display while the user drags
//  the minimized window.
HCURSOR CbcdemomfcguiDlg::OnQueryDragIcon()
{
	return static_cast<HCURSOR>(m_hIcon);
}



void CbcdemomfcguiDlg::OnBnClickedButtonInitBc()
{
	int wResult;
	S16BIT wSelection = 0x0000;
	S16BIT aOpCodes[10] = { 0x0000 };
	U32BIT u32MsgOptions = 1;

	/* Buffer containing data for initial values of data blocks */
	U16BIT wBuffer[32] =
	{
		0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
		0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
		0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
		0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004
	};

	/* Get user selected 1553 channel number (Logical Device Number) */
	m_devNum = m_comboBox.GetCurSel();

	/* Initialize the channel as a BC */
	wResult = aceInitialize(m_devNum, ACE_ACCESS_CARD, ACE_MODE_BC, 0, 0, 0);
	if (wResult)
	{
		MessageBox(_T("aceInitialize Failed"), NULL, MB_OK);
		return;
	}

	/* Create data blocks */
	aceBCDataBlkCreate(m_devNum, DBLK1, 32, wBuffer, 32);
	aceBCDataBlkCreate(m_devNum, DBLK2, 32, wBuffer, 32);

	/* Different data for the second BC->RT message */
	for (int i = 0; i < 32; i++)
	{
		wBuffer[i] = i;
	}

	aceBCDataBlkCreate(m_devNum, DBLK3, 32, wBuffer, 32);
	aceBCDataBlkCreate(m_devNum, DBLK4, 32, wBuffer, 32);
	
	/* Set Bus (A or B) on which message will be transmitted */
	if (m_rbBusA.GetCheck())
		u32MsgOptions = ACE_BCCTRL_CHL_A;
	else
		u32MsgOptions = ACE_BCCTRL_CHL_B;

	/* Set message retry if enabled and also set retry options */
	if (m_cbMsgRetry.GetCheck())
	{
		u32MsgOptions |= ACE_BCCTRL_RETRY_ENA;

		/* Set retry options */
		aceBCSetMsgRetry(m_devNum, ACE_RETRY_ONCE, ACE_RETRY_ALT, ACE_RETRY_ALT, 0);
	}

	/* Create message block */
	aceBCMsgCreateBCtoRT(
		m_devNum,              /* Device number                    */
		MSG1,                /* Message ID to create             */
		DBLK1,               /* Message will use this data block */
		gMsgPara[0].rtAddr,	 /* RT address                       */
		gMsgPara[0].subAddr, /* RT subaddress                    */
		gMsgPara[0].wordCnt, /* Word count                       */
		0,                   /* Default message timer            */
		u32MsgOptions);   /* use chl A options                */

	/* Create XEQ opcode that will use msg block */
	aceBCOpCodeCreate(m_devNum, OP1, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, MSG1, 0, 0);

	aceBCMsgCreateRTtoBC(
		m_devNum,              /* Device number                    */
		MSG2,                /* Message ID to create             */
		DBLK2,               /* Message will use this data block */
		gMsgPara[1].rtAddr,	 /* RT address                       */
		gMsgPara[1].subAddr, /* RT subaddress                    */
		gMsgPara[1].wordCnt, /* Word count                       */
		0,                   /* Default message timer            */
		u32MsgOptions);   /* use chl A options                */

	/* Create XEQ opcode that will use msg block */
	aceBCOpCodeCreate(m_devNum, OP2, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, MSG2, 0, 0);

	aceBCMsgCreateBCtoRT(
		m_devNum,              /* Device number                    */
		MSG3,                /* Message ID to create             */
		DBLK3,               /* Message will use this data block */
		gMsgPara[2].rtAddr,	 /* RT address                       */
		gMsgPara[2].subAddr, /* RT subaddress                    */
		gMsgPara[2].wordCnt, /* Word count                       */
		0,                   /* Default message timer            */
		u32MsgOptions);   /* use chl A options                */

	/* Create XEQ opcode that will use msg block */
	aceBCOpCodeCreate(m_devNum, OP3, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, MSG3, 0, 0);

	aceBCMsgCreateRTtoBC(
		m_devNum,              /* Device number                    */
		MSG4,                /* Message ID to create             */
		DBLK4,               /* Message will use this data block */
		gMsgPara[3].rtAddr,	 /* RT address                       */
		gMsgPara[3].subAddr, /* RT subaddress                    */
		gMsgPara[3].wordCnt, /* Word count                       */
		0,                   /* Default message timer            */
		u32MsgOptions);   /* use chl A options                */

	/* Create XEQ opcode that will use msg block */
	aceBCOpCodeCreate(m_devNum, OP4, ACE_OPCODE_XEQ, ACE_CNDTST_ALWAYS, MSG4, 0, 0);

	/* Create CAL opcode that will call mnr frame from major */
	aceBCOpCodeCreate(m_devNum, OPCAL1, ACE_OPCODE_CAL, ACE_CNDTST_ALWAYS, MNR1, 0, 0);
	aceBCOpCodeCreate(m_devNum, OPCAL2, ACE_OPCODE_CAL, ACE_CNDTST_ALWAYS, MNR2, 0, 0);

	/* Create Minor Frame 1 */
	aOpCodes[0] = OP1;
	aOpCodes[1] = OP2;
	aceBCFrameCreate(m_devNum, MNR1, ACE_FRAME_MINOR, aOpCodes, 2, 0, 0);

	/* Create Minor Frame 1 */
	aOpCodes[0] = OP3;
	aOpCodes[1] = OP4;
	aceBCFrameCreate(m_devNum, MNR2, ACE_FRAME_MINOR, aOpCodes, 2, 0, 0);

	/* Create Major Frame */
	aOpCodes[0] = OPCAL1;
	aOpCodes[1] = OPCAL2;
	aceBCFrameCreate(m_devNum, MJR, ACE_FRAME_MAJOR, aOpCodes, 2, 1000, 0);

	/* Create Host Buffer */
	aceBCInstallHBuf(m_devNum, 32 * 1024);

	m_btnStartBC.EnableWindow(true);
	m_btnFreeBC.EnableWindow(true);
	m_btnInitializeBC.EnableWindow(false);
	m_btnStopBC.EnableWindow(false);
	m_btnWrite.EnableWindow(false);

	m_rbBusA.EnableWindow(false);
	m_rbBusB.EnableWindow(false);
	m_cbMsgRetry.EnableWindow(false);
}

void CbcdemomfcguiDlg::CreateDisplayString
(
	MSGSTRUCT* pMsg,
	CString* pStr
)
{
	U16BIT i;
	char szBuffer[100];
	U16BIT wRT, wTR1, wTR2, wSA, wWC;
	CString tmpStr;

	/* Display message header info */
	tmpStr.Format(_T("TIME = %08dus    BUS %c   TYPE%d: %s "),
		pMsg->wTimeTag * 2, pMsg->wBlkSts & ACE_BC_BSW_CHNL ? 'B' : 'A',
		pMsg->wType, aceGetMsgTypeString(pMsg->wType));
	*pStr += tmpStr;

	/* Display command word info */
	aceCmdWordParse(pMsg->wCmdWrd1, &wRT, &wTR1, &wSA, &wWC);
	sprintf_s(szBuffer, "%02d-%c-%02d-%02d", wRT, wTR1 ? 'T' : 'R', wSA, wWC);
	tmpStr.Format(_T("\n            CMD1 %04X --> %s"), pMsg->wCmdWrd1, szBuffer);
	*pStr += tmpStr;

	tmpStr.Format(_T("\tBSW %04X "), pMsg->wBlkSts);
	*pStr += tmpStr;

	if (pMsg->wCmdWrd2Flg)
	{
		aceCmdWordParse(pMsg->wCmdWrd2, &wRT, &wTR2, &wSA, &wWC);
		sprintf_s(szBuffer, "%02d-%c-%02d-%02d", wRT, wTR2 ? 'T' : 'R', wSA, wWC);
		tmpStr.Format(_T("\n            CMD2 %04X --> %s"), pMsg->wCmdWrd2, szBuffer);
		*pStr += tmpStr;
	}

	/* Display transmit status words */
	if ((wTR1 == 1) || (pMsg->wBCCtrlWrd & ACE_BCCTRL_RT_TO_RT))
	{
		if (pMsg->wStsWrd1Flg)
		{
			tmpStr.Format(_T("\n            STA1 %04X"), pMsg->wStsWrd1);
			*pStr += tmpStr;
		}
	}

	/* Display Data words */
	for (i = 0; i < pMsg->wWordCount; i++)
	{
		if (i == 0)
		{
			tmpStr.Format(_T("\n            DATA "));
			*pStr += tmpStr;
		}

		tmpStr.Format(_T("%04X  "), pMsg->aDataWrds[i]);
		*pStr += tmpStr;

		if ((i % 8) == 7)
		{
			tmpStr.Format(_T("\n                 "));
			*pStr += tmpStr;
		}
	}

	/* Display receive status words */
	if ((wTR1 == 0) && !(pMsg->wBCCtrlWrd & ACE_BCCTRL_RT_TO_RT))
	{
		if (pMsg->wStsWrd1Flg)
		{
			tmpStr.Format(_T("\n            STA1 %04X"), pMsg->wStsWrd1);
			*pStr += tmpStr;
		}
	}

	if (pMsg->wStsWrd2Flg)
	{
		tmpStr.Format(_T("\n            STA2 %04X"), pMsg->wStsWrd2);
		*pStr += tmpStr;
	}

	/* Display Error information */
	if (pMsg->wBlkSts & 0x170f)
	{
		tmpStr.Format(_T("\n ERROR: %s"), aceGetBSWErrString(ACE_MODE_BC, pMsg->wBlkSts));
		*pStr += tmpStr;
	}
}

void CbcdemomfcguiDlg::OnBnClickedButtonStartBc()
{
	int wResult;
	int wRepeatCount = -1;

	quitThread = FALSE;

	/* Start BC */
	wResult = aceBCStart(m_devNum, MJR, wRepeatCount);
	if (wResult)
	{
		char strBuf[250];
		aceErrorStr(wResult, strBuf, 250);
		CString msg;
		msg.Format(_T("BCStart Failed: %s"), strBuf);
		MessageBox(msg, NULL, MB_OK);
		return;
	}

	/* Change state of buttons */
	m_btnStopBC.EnableWindow(true);
	m_btnWrite.EnableWindow(true);
	m_btnStartBC.EnableWindow(false);
	m_btnFreeBC.EnableWindow(false);

	/* Create a thread to read BC host buffer msgs */
	hThreadMsgGet = CreateThread(NULL,			/* default security attributes  */
		0,										/* use default stack size       */
		(LPTHREAD_START_ROUTINE)ThreadMsgGet,	/* thread function   */
		(LPVOID)this,							/* no thread function argument  */
		0,										/* use default creation flags   */
		&dwIdThreadMsgGet);

	if (hThreadMsgGet == NULL)
	{
		AfxMessageBox(_T("CreateThread for hThreadMsgGet Failed!"));
	}
}

/*******************************************************************************
 * Name:    OnNewHbufMsgAvailable
 *
 * Description:
 *
 *      This is a user-defined windows message handler for the message 
 * WM_PROCESS_NEW_HBUF_MSG_AVLBL. This message is posted by ThreadMsgGet
 * when it finds a message in the BC Host Buffer. wParam contains the 
 * message number (out of Msg 1 to Msg 4).
 * 
 * The handler processes the received message for user-friendly display 
 * in the respective rich edit control 
 *
 * In   lpParam: A parameter passed when the thread was created
 * Out  none
 ******************************************************************************/
LRESULT	CbcdemomfcguiDlg::OnNewHbufMsgAvailable(WPARAM wParam, LPARAM lParam)
{
	//UpdateData(true);
	CString displayStr = "";

	/* Update display with data from the received message */
	switch ((int)wParam)	/* wParam contains the message number */
	{
	case 0:
		m_rtbRawMsgDisplay_1.SetWindowTextA(_T(""));
		CreateDisplayString(&gMsgs[(int)wParam], &displayStr);
		m_rtbRawMsgDisplay_1.SetWindowTextA(displayStr);
		break;
	case 1:
		m_rtbRawMsgDisplay_2.SetWindowTextA(_T(""));
		CreateDisplayString(&gMsgs[(int)wParam], &displayStr);
		m_rtbRawMsgDisplay_2.SetWindowTextA(displayStr);
		break;
	case 2:
		m_rtbRawMsgDisplay_3.SetWindowTextA(_T(""));
		CreateDisplayString(&gMsgs[(int)wParam], &displayStr);
		m_rtbRawMsgDisplay_3.SetWindowTextA(displayStr);
		break;
	case 3:
		m_rtbRawMsgDisplay_4.SetWindowTextA(_T(""));
		CreateDisplayString(&gMsgs[(int)wParam], &displayStr);
		m_rtbRawMsgDisplay_4.SetWindowTextA(displayStr);
		break;
	}

	//UpdateData(false);
	return 0;
}

/*******************************************************************************
 * Name:    ThreadMsgGet
 *
 * Description:
 *
 *      A separate thread to read BC host buffer
 *
 * In   lpParam: A parameter passed when the thread was created
 * Out  none
 ******************************************************************************/
DWORD WINAPI ThreadMsgGet(LPVOID lpParam)
{
	CbcdemomfcguiDlg* pDlg = (CbcdemomfcguiDlg*)lpParam;
	S16BIT nResult = 0x0000;
	U32BIT dwMsgCount = 0x00000000;
	MSGSTRUCT sMsg;

	while (!quitThread)
	{
		/* Check host buffer for msgs */
		nResult = aceBCGetHBufMsgDecoded(pDlg->m_devNum, &sMsg, &dwMsgCount, &dwHBufLost, ACE_BC_MSGLOC_NEXT_PURGE);

		if (nResult)
		{
			/* Error reading HBuf */
			continue;
		}

		/* Message found */
		if (dwMsgCount)
		{
			/* Update total messages counter */
			++dwCurCount;

			/* Analyze the command word to identify this message */
			U16BIT RT = 0, TR = 0, SA = 0, WC = 0;
			nResult = aceCmdWordParse(sMsg.wCmdWrd1, &RT, &TR, &SA, &WC);

			/* Based on the message identified, save MSGSTRUCT and notify GUI */
			for (int i = 0; i < NO_OF_MSGS; i++)
			{
				if (RT == gMsgPara[i].rtAddr && TR == gMsgPara[i].TR && SA == gMsgPara[i].subAddr && WC == gMsgPara[i].wordCnt)	// MSG1
				{
					/* Save message into a global structure array */
					gMsgs[i] = sMsg;

					/* Notify GUI thread */
					pDlg->PostMessage(WM_PROCESS_NEW_HBUF_MSG_AVLBL, i);
				}
			}
		}

		Sleep(1);
	}
	return 0;
}


void CbcdemomfcguiDlg::OnBnClickedButtonStopBc()
{
	int wResult;

	/* Stop Thread */
	quitThread = TRUE;

	/* Wait for a while to allow the thread to exit */
	Sleep(100);

	/* Stop BC */
	wResult = aceBCStop(m_devNum);
	if (wResult)
	{
		MessageBox(_T("BCStop Failed"), NULL, MB_OK);
		return;
	}

	/* Change state of buttons */
	m_btnStopBC.EnableWindow(false);
	m_btnStartBC.EnableWindow(true);
	m_btnFreeBC.EnableWindow(true);
	m_btnWrite.EnableWindow(false);
}


void CbcdemomfcguiDlg::OnBnClickedButtonFreeBc()
{
	/* Free the card */
	S16BIT wResult = aceFree(m_devNum);
	if (wResult)
	{
		MessageBox(_T("aceFree Failed"), NULL, MB_OK);
		return;
	}

	/* Change state of buttons */
	m_btnFreeBC.EnableWindow(false);
	m_btnInitializeBC.EnableWindow(true);
	m_btnStopBC.EnableWindow(false);
	m_btnStartBC.EnableWindow(false);

	m_rbBusA.EnableWindow(true);
	m_rbBusB.EnableWindow(true);
	m_cbMsgRetry.EnableWindow(true);
}


void CbcdemomfcguiDlg::OnBnClickedCancel()
{
	// Check if BC is running. If yes, abort with a message
	if (!m_btnInitializeBC.IsWindowEnabled())
	{
		MessageBox(_T("Cannot close application while channel is in use. Free the channel and retry."), NULL, MB_OK);
	}
	else
	{
		CDialogEx::OnCancel();
	}
}


void CbcdemomfcguiDlg::OnBnClickedButtonWrite()
{
	U16BIT wBuffer[32];
	S16BIT dblkId = -1;

	// Get current msg selection
	int msgSelected = m_comboMsgSelect.GetCurSel();

	// Choose the correct data block to write based on the message selection
	dblkId = (msgSelected == 0) ? DBLK1 : DBLK3;

	// Validate the data value entered by the user 
	UpdateData(TRUE);
	if ((m_DataToWrite < 0) || m_DataToWrite > 0xFFFF) // Valid values are 0 to 65535
	{
		MessageBox(_T("Value out of range!"), NULL, MB_OK);
		m_editDataToWrite.SetFocus();
		return;
	}

	// Populate a 32 word buffer with data input by the user
	for(int i = 0; i < 32; i++)
	{
		wBuffer[i] = (U16BIT)m_DataToWrite;
	}

	// call aceBCDataBlkWrite to write the data to the data block
	S16BIT wResult = aceBCDataBlkWrite(m_devNum, dblkId, wBuffer, 32, 0);
	if (wResult)
	{
		MessageBox(_T("aceBCDataBlkWrite Failed"), NULL, MB_OK);
		return;
	}
}
