/******************************************************************************
 *
 * RTips Technologies, Bangalore (www.rtipsonline.com)
 *
 * This sample demonstrates using DDC's AceXtreme SDK in a Qt GUI application.
 *
 * Many portions of this file are adapted from the rtpoll.c sample provided in
 * the AceXtreme SDK (BU-69092Sx) of M/s. Data Device Corporation
 *
 * This sample demonstrates using DDC's AceXtreme SDK in a Qt GUI application.
 * The sample runs the Enhanced-Mini-ACE in RT mode. Using simple polling, the stack
 * is queried for messages and read off as found. It displays the messages in a rich text box.
 * It allows the user to specify the RT address and sub addresses. Lastly, it demonstrates the
 * aceRTDataBlkWrite API to update data.
 *	API's demonstrated in this sample
 *	aceInitialize
 *  aceRTDataBlkCreate
 *  aceRTSetAddress
 *  aceRTDataBlkMapToSA
 *  aceRTMsgLegalityEnable
 *  aceRTGetAddress
 *	aceRTStart
 *	aceRTGetStkMsgDecoded
 *	aceCmdWordParse
 *	aceRTStop
 *	aceFree
 *  aceErrorStr
 *  aceGetMsgTypeString
 *  aceGetBSWErrString
 *  aceRTDataBlkWrite
 *
 * Author: Subhashini B S
 * Last updated: Feb 08, 2025
 ******************************************************************************/

#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QCloseEvent>
#include <QMessageBox>

MainWindow::MainWindow(QWidget *parent)
    : QMainWindow(parent)
    , ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    /* Connect the signal emitted by msgGetThread to a slot */
    QObject::connect(&m_msgGetThread,
                     SIGNAL(NewHbufMsgAvlbl(int,MSGSTRUCT)),
                     this,
                     SLOT(ProcessNewHbufMsgAvlbl(int,MSGSTRUCT)));

    ui->pushButton_FreeChannel->setEnabled(false);
    ui->pushButton_InitRT->setEnabled(true);
    ui->pushButton_StopRT->setEnabled(false);
    ui->pushButton_StartRT->setEnabled(false);
    ui->pushButton_Write->setEnabled(false);

    QListWidgetItem *item, *txitem;
    for (int saNum=0; saNum<32; saNum++)
    {
        item = new QListWidgetItem();
        item->setData(Qt::DisplayRole, saNum);
        ui->listWidget_AvlblSA->addItem(item);

        txitem = new QListWidgetItem();
        txitem->setData(Qt::DisplayRole, saNum);
        ui->listWidget_AvlblTxSA->addItem(txitem);
    }
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    // Check if RT is running. If yes, abort with a message
    if(!ui->pushButton_InitRT->isEnabled())
    {
        QMessageBox::critical(this, "Close", "Cannot close application while channel is in use. Free the channel and retry.");
        event->ignore();
    }
}


void MainWindow::on_pushButton_InitRT_clicked()
{
    /* Variable Definition */
    S16BIT wResult = 0x0000;
    /* Buffer containing data for initial values of data blocks */
/*	U16BIT wBuffer[32] =
    {
        0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
        0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
        0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004,
        0x0001, 0x0002, 0x0003, 0x0004, 0x0001, 0x0002, 0x0003, 0x0004
    };*/
    U16BIT wBufferRx[32][32];	// 32 buffers for Rx SAs
    U16BIT wBufferTx[32][32];	// 32 buffers for Tx SAs
    int i, j;
    int numRxSA, numTxSA;

    numRxSA = ui->listWidget_UsedSA->count();
    if (numRxSA == 0)
    {
        QMessageBox::critical(this, "Init RT", "Atleast one Rx SubAddress should be used!");
        return;
    }

    numTxSA =  ui->listWidget_UsedTxSA->count();
    if (numTxSA == 0)
    {
        QMessageBox::critical(this, "Init RT", "Atleast one Tx SubAddress should be used!");
        return;
    }

    for (i = 0; i < 32; i++)
    {
        for (j = 0; j < 32; j++)
        {
            wBufferRx[i][j] = j;
            wBufferTx[i][j] = j;
        }
    }

    /* Get user selected 1553 channel number (Logical Device Number) */
    m_devNum = ui->comboBox_LDN->currentIndex();

    /* Initialize the channel as a RT */
    wResult = aceInitialize(m_devNum, ACE_ACCESS_CARD, ACE_MODE_RT, 0, 0, 0);
    if (wResult)
    {
        QMessageBox::critical(this, "Init RT", "aceInitialize Failed");
        return;
    }

    /* Get RT Address */
    m_RTAddr = ui->comboBox_RTAddr->currentIndex();

    /* Create DBLKs for RT */
    for (i = 0; i < 32; i++)
    {
        aceRTDataBlkCreate(m_devNum, RX_DBLK_BASE + i, ACE_RT_DBLK_SINGLE, wBufferRx[i], 32);
        aceRTDataBlkCreate(m_devNum, TX_DBLK_BASE + i, ACE_RT_DBLK_SINGLE, wBufferTx[i], 32);
    }

    /* Set RT [or latch (BU-65565 only)] address */
    aceRTSetAddress(m_devNum, m_RTAddr);

    int intSA;
    for (int saCtr = 0; saCtr < numRxSA; saCtr++)
    {
        intSA = ui->listWidget_UsedSA->item(saCtr)->text().toInt();

        //QMessageBox::information(this, "Init RT, Rx SA = ", QString::number(intSA));

        /* Map data block to given Rx sub-address */
        aceRTDataBlkMapToSA(m_devNum, RX_DBLK_BASE + intSA, intSA, ACE_RT_MSGTYPE_RX, 0, TRUE);

        /*
           aceRTMsgLegalityEnable function has been added
           in order to receive a proper response from any
           RT address configured.
        */
        aceRTMsgLegalityEnable(m_devNum, ACE_RT_OWN_ADDRESS, 0 /* 0 for receive */, intSA, 0xFFFFFFFF);
    }

    for (int saCtr = 0; saCtr < numTxSA; saCtr++)
    {
        intSA = ui->listWidget_UsedTxSA->item(saCtr)->text().toInt();

        //QMessageBox::information(this, "Init RT, Tx SA = ", QString::number(intSA));

        /* Map data block to given Tx sub-address */
        aceRTDataBlkMapToSA(m_devNum, TX_DBLK_BASE + intSA, intSA, ACE_RT_MSGTYPE_TX, 0, TRUE);

        /*
           aceRTMsgLegalityEnable function has been added
           in order to receive a proper response from any
           RT address configured.
        */
        aceRTMsgLegalityEnable(m_devNum, ACE_RT_OWN_ADDRESS, 1 /* 1 for transmit */, intSA, 0xFFFFFFFF);
    }

    /* Check that RT Address was set */
    aceRTGetAddress(m_devNum, (U16BIT *) &m_RTAddr);

    ui->pushButton_StartRT->setEnabled(true);
    ui->pushButton_FreeChannel->setEnabled(true);
    ui->pushButton_InitRT->setEnabled(false);
    ui->pushButton_AddSA->setEnabled(false);
    ui->pushButton_RemoveSA->setEnabled(false);
    ui->pushButton_AddTxSA->setEnabled(false);
    ui->pushButton_RemoveTxSA->setEnabled(false);
    ui->pushButton_StopRT->setEnabled(false);
    ui->pushButton_Write->setEnabled(false);
}


void MainWindow::on_pushButton_StartRT_clicked()
{
    int wResult;

    /* Start RT Device */
    wResult = aceRTStart(m_devNum);
    if (wResult)
    {
        char strBuf[250];
        aceErrorStr(wResult, strBuf, 250);
        QMessageBox::critical(this, "Start RT", "RTStart Failed: " + QString::fromUtf8(strBuf));
        return;
    }

    /* Change state of buttons */
    ui->pushButton_StopRT->setEnabled(true);
    ui->pushButton_Write->setEnabled(true);
    ui->pushButton_StartRT->setEnabled(false);
    ui->pushButton_FreeChannel->setEnabled(false);

    // Clear the message display text edit
    ui->textEdit_Msg->clear();

    /* Create a thread to read RT msgs */
    m_msgGetThread.SetPtr(this);
    m_msgGetThread.stop = false;
    m_msgGetThread.start();
}


void MainWindow::on_pushButton_StopRT_clicked()
{
    int wResult;

    /* Stop Thread */
    m_msgGetThread.stop = true;
    /* Wait to allow the thread to exit */
    while(m_msgGetThread.IsThreadStopped()==false);

    /* Stop RT Device */
    wResult = aceRTStop(m_devNum);
    if (wResult)
    {
        QMessageBox::critical(this, "Stop RT", "RTStop Failed");
        return;
    }

    /* Change state of buttons */
    ui->pushButton_StopRT->setEnabled(false);
    ui->pushButton_StartRT->setEnabled(true);
    ui->pushButton_FreeChannel->setEnabled(true);
    ui->pushButton_Write->setEnabled(false);
}


void MainWindow::on_pushButton_FreeChannel_clicked()
{
    /* Free Ace Memory and Device */
    S16BIT wResult = aceFree(m_devNum);
    if (wResult)
    {
        QMessageBox::critical(this, "Free Channel", "aceFree Failed");
        return;
    }

    /* Change state of buttons */
    ui->pushButton_FreeChannel->setEnabled(false);
    ui->pushButton_InitRT->setEnabled(true);
    ui->pushButton_AddSA->setEnabled(true);
    ui->pushButton_RemoveSA->setEnabled(true);
    ui->pushButton_AddTxSA->setEnabled(true);
    ui->pushButton_RemoveTxSA->setEnabled(true);
    ui->pushButton_StopRT->setEnabled(false);
    ui->pushButton_StartRT->setEnabled(false);
}

void MainWindow::ProcessNewHbufMsgAvlbl(int msgNum, MSGSTRUCT strMsg)
{
    QString displayStr;

    /* Update display with data from the received message */
    displayStr = CreateDisplayString(msgNum, &strMsg);
    ui->textEdit_Msg->moveCursor(QTextCursor::End);
    ui->textEdit_Msg->insertPlainText(displayStr);
}

QString MainWindow::CreateDisplayString(int msgNum, MSGSTRUCT *pMsg)
{
    U16BIT i;
    char szBuffer[100];
    U16BIT wRT, wTR1, wTR2, wSA, wWC;
    QString tmpStr, retStr;

    /* Display message header info */
    tmpStr = "\nMSG #" + QString::number(msgNum).rightJustified(8, '0') + ".  TIME = " + QString::number(pMsg->wTimeTag * 2).rightJustified(8, '0') + "us    BUS " +
            (pMsg->wBlkSts & ACE_BC_BSW_CHNL ? "B" : "A") + "   TYPE" + QString::number(pMsg->wType) + ": " + QString::fromUtf8(aceGetMsgTypeString(pMsg->wType)) + " ";
    retStr += tmpStr;

    /* Display command word info */
    aceCmdWordParse(pMsg->wCmdWrd1, &wRT, &wTR1, &wSA, &wWC);
    sprintf_s(szBuffer, "%02d-%c-%02d-%02d", wRT, wTR1 ? 'T' : 'R', wSA, wWC);
    tmpStr = "\n            CMD1 " + QString::number(pMsg->wCmdWrd1, 16).rightJustified(4, '0').toUpper() + " --> " + QString::fromUtf8(szBuffer);
    retStr += tmpStr;

    tmpStr = "\tBSW " + QString::number(pMsg->wBlkSts, 16).rightJustified(4, '0').toUpper() + " ";
    retStr += tmpStr;

    if (pMsg->wCmdWrd2Flg)
    {
        aceCmdWordParse(pMsg->wCmdWrd2, &wRT, &wTR2, &wSA, &wWC);
        sprintf_s(szBuffer, "%02d-%c-%02d-%02d", wRT, wTR2 ? 'T' : 'R', wSA, wWC);
        tmpStr = "\n            CMD2 " + QString::number(pMsg->wCmdWrd2, 16).rightJustified(4, '0').toUpper() + " --> " + QString::fromUtf8(szBuffer);
        retStr += tmpStr;
    }

    /* Display transmit status words */
    if ((wTR1 == 1) || (pMsg->wBlkSts & ACE_MT_BSW_RTRT))
    {
        if (pMsg->wStsWrd1Flg)
        {
            tmpStr = "\n            STA1 " + QString::number(pMsg->wStsWrd1, 16).rightJustified(4, '0').toUpper();
            retStr += tmpStr;
        }
    }

    /* Display Data words */
    for (i = 0; i < pMsg->wWordCount; i++)
    {
        if (i == 0)
        {
            tmpStr = "\n            DATA ";
            retStr += tmpStr;
        }

        tmpStr = QString::number(pMsg->aDataWrds[i], 16).rightJustified(4, '0').toUpper() + "  ";
        retStr += tmpStr;

        if ((i % 8) == 7)
        {
            tmpStr = "\n                 ";
            retStr += tmpStr;
        }
    }

    /* Display receive status words */
    if ((wTR1 == 0) && !(pMsg->wBlkSts & ACE_MT_BSW_RTRT))
    {
        if (pMsg->wStsWrd1Flg)
        {
            tmpStr = "\n            STA1 " + QString::number(pMsg->wStsWrd1, 16).rightJustified(4, '0').toUpper();
            retStr += tmpStr;
        }
    }

    if (pMsg->wStsWrd2Flg)
    {
        tmpStr = "\n            STA2 " + QString::number(pMsg->wStsWrd2, 16).rightJustified(4, '0').toUpper();
        retStr += tmpStr;
    }

    /* Display Error information */
    if (pMsg->wBlkSts & ACE_MT_BSW_ERRFLG)
    {
        tmpStr = "\n ERROR: " + QString::fromUtf8(aceGetBSWErrString(ACE_MODE_MT, pMsg->wBlkSts)) + "\n\n";
        retStr += tmpStr;
    }

    return retStr;
}


void MainWindow::on_pushButton_AddSA_clicked()
{
    if(ui->listWidget_AvlblSA->currentItem())
    {
        QListWidgetItem *item = new QListWidgetItem();
        item->setData(Qt::DisplayRole, ui->listWidget_AvlblSA->currentItem()->text().toInt());
        ui->listWidget_UsedSA->addItem(item);
        ui->listWidget_UsedSA->sortItems();

        delete ui->listWidget_AvlblSA->takeItem(ui->listWidget_AvlblSA->currentRow());
        ui->listWidget_AvlblSA->sortItems();
    }
    else
    {
        QMessageBox::critical(this, "Add Rx SA", "Select a SubAddress from the list of 'Available SubAddresses'!");
    }
}


void MainWindow::on_pushButton_RemoveSA_clicked()
{
    if(ui->listWidget_UsedSA->currentItem())
    {
        QListWidgetItem *item = new QListWidgetItem();
        item->setData(Qt::DisplayRole, ui->listWidget_UsedSA->currentItem()->text().toInt());
        ui->listWidget_AvlblSA->addItem(item);
        ui->listWidget_AvlblSA->sortItems();

        delete ui->listWidget_UsedSA->takeItem(ui->listWidget_UsedSA->currentRow());
        ui->listWidget_UsedSA->sortItems();
    }
    else
    {
        QMessageBox::critical(this, "Remove Rx SA", "Select a SubAddress from the list of 'Used SubAddresses'!");
    }
}

void MainWindow::on_pushButton_AddTxSA_clicked()
{
    if(ui->listWidget_AvlblTxSA->currentItem())
    {
        QListWidgetItem *item = new QListWidgetItem();
        item->setData(Qt::DisplayRole, ui->listWidget_AvlblTxSA->currentItem()->text().toInt());
        ui->listWidget_UsedTxSA->addItem(item);
        ui->listWidget_UsedTxSA->sortItems();

        delete ui->listWidget_AvlblTxSA->takeItem(ui->listWidget_AvlblTxSA->currentRow());
        ui->listWidget_AvlblTxSA->sortItems();
    }
    else
    {
        QMessageBox::critical(this, "Add Tx SA", "Select a SubAddress from the list of 'Available SubAddresses'!");
    }
}


void MainWindow::on_pushButton_RemoveTxSA_clicked()
{
    if(ui->listWidget_UsedTxSA->currentItem())
    {
        QListWidgetItem *item = new QListWidgetItem();
        item->setData(Qt::DisplayRole, ui->listWidget_UsedTxSA->currentItem()->text().toInt());
        ui->listWidget_AvlblTxSA->addItem(item);
        ui->listWidget_AvlblTxSA->sortItems();

        delete ui->listWidget_UsedTxSA->takeItem(ui->listWidget_UsedTxSA->currentRow());
        ui->listWidget_UsedTxSA->sortItems();
    }
    else
    {
        QMessageBox::critical(this, "Remove Tx SA", "Select a SubAddress from the list of 'Used SubAddresses'!");
    }
}

void MainWindow::on_pushButton_Close_clicked()
{
    // Check if RT is running. If yes, abort with a message
    if(!ui->pushButton_InitRT->isEnabled())
    {
        QMessageBox::critical(this, "Close", "Cannot close application while channel is in use. Free the channel and retry.");
    }
    else
    {
        this->close();
    }
}


void MainWindow::on_pushButton_Write_clicked()
{
    U16BIT wBuffer[32];

    // Validate the data value entered by the user
    if ((ui->lineEdit_DataToWrite->text().toInt() < 0) || ui->lineEdit_DataToWrite->text().toInt() > 0xFFFF) // Valid values are 0 to 65535
    {
        QMessageBox::critical(this, "Write", "Value out of range!");
        return;
    }

    int intSA;
    if(ui->listWidget_UsedTxSA->currentItem())
    {
        intSA = ui->listWidget_UsedTxSA->currentItem()->text().toInt();
    }
    else
    {
        QMessageBox::critical(this, "Write", "Select a Tx SubAddress from the list of 'Used SubAddresses'!");
    }

    // Populate a 32 word buffer with data input by the user
    for(int i = 0; i < 32; i++)
    {
        wBuffer[i] = (U16BIT)ui->lineEdit_DataToWrite->text().toInt();
    }

    // call aceRTDataBlkWrite to write the data to the data block
    S16BIT wResult = aceRTDataBlkWrite(m_devNum, TX_DBLK_BASE + intSA, wBuffer, 32, 0);
    if (wResult)
    {
        QMessageBox::critical(this, "Write", "aceRTDataBlkWrite Failed");
        return;
    }
}
